"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useEntityAnalyticsRoutes = void 0;
var _react = require("react");
var _constants = require("../../../common/constants");
var _kibana_react = require("../../common/lib/kibana/kibana_react");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * This hardcoded name was temporarily introduced for 9.1.0.
 * It is used to identify the only entity source that can be edited by the UI.
 */
const ENTITY_SOURCE_NAME = 'User Monitored Indices';
const useEntityAnalyticsRoutes = () => {
  const http = (0, _kibana_react.useKibana)().services.http;
  return (0, _react.useMemo)(() => {
    /**
     * Fetches preview risks scores
     */
    const fetchRiskScorePreview = ({
      signal,
      params
    }) => http.fetch(_constants.RISK_SCORE_PREVIEW_URL, {
      version: '1',
      method: 'POST',
      body: JSON.stringify(params),
      signal
    });

    /**
     * Fetches entities from the Entity Store
     */
    const fetchEntitiesList = ({
      signal,
      params
    }) => http.fetch(_constants.LIST_ENTITIES_URL, {
      version: _constants.API_VERSIONS.public.v1,
      method: 'GET',
      query: {
        entity_types: params.entityTypes,
        sort_field: params.sortField,
        sort_order: params.sortOrder,
        page: params.page,
        per_page: params.perPage,
        filterQuery: params.filterQuery
      },
      signal
    });

    /**
     * Fetches risks engine status
     */
    const fetchRiskEngineStatus = ({
      signal
    }) => http.fetch(_constants.RISK_ENGINE_STATUS_URL, {
      version: '1',
      method: 'GET',
      signal
    });

    /**
     * Init risk score engine
     */
    const initRiskEngine = () => http.fetch(_constants.RISK_ENGINE_INIT_URL, {
      version: '1',
      method: 'POST'
    });

    /**
     * Enable risk score engine
     */
    const enableRiskEngine = () => http.fetch(_constants.RISK_ENGINE_ENABLE_URL, {
      version: '1',
      method: 'POST'
    });

    /**
     * Disable risk score engine
     */
    const disableRiskEngine = () => http.fetch(_constants.RISK_ENGINE_DISABLE_URL, {
      version: '1',
      method: 'POST'
    });

    /**
     * Enable risk score engine
     */
    const scheduleNowRiskEngine = () => http.fetch(_constants.RISK_ENGINE_SCHEDULE_NOW_URL, {
      version: _constants.API_VERSIONS.public.v1,
      method: 'POST'
    });

    /**
     * Calculate and stores risk score for an entity
     */
    const calculateEntityRiskScore = params => {
      return http.fetch(_constants.RISK_SCORE_ENTITY_CALCULATION_URL, {
        version: '1',
        method: 'POST',
        body: JSON.stringify(params)
      });
    };

    /**
     * Get risk engine privileges
     */
    const fetchRiskEnginePrivileges = () => http.fetch(_constants.RISK_ENGINE_PRIVILEGES_URL, {
      version: '1',
      method: 'GET'
    });

    /**
     * Get asset criticality privileges
     */
    const fetchAssetCriticalityPrivileges = () => http.fetch(_constants.ASSET_CRITICALITY_INTERNAL_PRIVILEGES_URL, {
      version: '1',
      method: 'GET'
    });

    /**
     * Get Entity Store privileges
     */
    const fetchEntityStorePrivileges = () => http.fetch(_constants.ENTITY_STORE_INTERNAL_PRIVILEGES_URL, {
      version: '1',
      method: 'GET'
    });

    /**
     * Search indices for privilege monitoring import
     */
    const searchPrivMonIndices = async params => http.fetch(_constants.PRIVMON_INDICES_URL, {
      version: _constants.API_VERSIONS.public.v1,
      method: 'GET',
      query: {
        searchQuery: params.query
      },
      signal: params.signal
    });

    /**
     * Create an index for privilege monitoring import
     */
    const createPrivMonImportIndex = async params => http.fetch(_constants.PRIVMON_INDICES_URL, {
      version: _constants.API_VERSIONS.public.v1,
      method: 'PUT',
      body: JSON.stringify({
        name: params.name,
        mode: params.mode
      }),
      signal: params.signal
    });
    /**
     * Register a data source for privilege monitoring engine
     */
    const registerPrivMonMonitoredIndices = async indexPattern => http.fetch(_constants.MONITORING_ENTITY_SOURCE_URL, {
      version: _constants.API_VERSIONS.public.v1,
      method: 'POST',
      body: JSON.stringify({
        type: 'index',
        name: ENTITY_SOURCE_NAME,
        indexPattern
      })
    });

    /**
     * Update a data source for privilege monitoring engine
     */
    const updatePrivMonMonitoredIndices = async (id, indexPattern) => http.fetch((0, _constants.getPrivmonMonitoringSourceByIdUrl)(id), {
      version: _constants.API_VERSIONS.public.v1,
      method: 'PUT',
      body: JSON.stringify({
        type: 'index',
        name: ENTITY_SOURCE_NAME,
        indexPattern
      })
    });

    /**
     * Create asset criticality
    /**
     *
     *
     * @param {(Pick<AssetCriticality, 'idField' | 'idValue' | 'criticalityLevel'> & {
     *         refresh?: 'wait_for';
     *       })} params
     * @return {*}  {Promise<AssetCriticalityRecord>}
     */
    const createAssetCriticality = async params => http.fetch(_constants.ASSET_CRITICALITY_PUBLIC_URL, {
      version: _constants.API_VERSIONS.public.v1,
      method: 'POST',
      body: JSON.stringify({
        id_value: params.idValue,
        id_field: params.idField,
        criticality_level: params.criticalityLevel,
        refresh: params.refresh
      })
    });
    const deleteAssetCriticality = async params => {
      await http.fetch(_constants.ASSET_CRITICALITY_PUBLIC_URL, {
        version: _constants.API_VERSIONS.public.v1,
        method: 'DELETE',
        query: {
          id_value: params.idValue,
          id_field: params.idField,
          refresh: params.refresh
        }
      });

      // spoof a response to allow us to better distnguish a delete from a create in use_asset_criticality.ts
      return {
        deleted: true
      };
    };

    /**
     * Get asset criticality
     */
    const fetchAssetCriticality = async params => {
      return http.fetch(_constants.ASSET_CRITICALITY_PUBLIC_URL, {
        version: _constants.API_VERSIONS.public.v1,
        method: 'GET',
        query: {
          id_value: params.idValue,
          id_field: params.idField
        }
      });
    };

    /**
     * Get multiple asset criticality records
     */
    const fetchAssetCriticalityList = async params => {
      const wrapWithQuotes = each => `"${each}"`;
      const kueryValues = `${params.idValues.map(wrapWithQuotes).join(' OR ')}`;
      const kuery = `${params.idField}: (${kueryValues})`;
      return http.fetch(_constants.ASSET_CRITICALITY_PUBLIC_LIST_URL, {
        version: _constants.API_VERSIONS.public.v1,
        method: 'GET',
        query: {
          kuery
        }
      });
    };
    const uploadAssetCriticalityFile = async (fileContent, fileName) => {
      const file = new File([new Blob([fileContent])], fileName, {
        type: 'text/csv'
      });
      const body = new FormData();
      body.append('file', file);
      return http.fetch(_constants.ASSET_CRITICALITY_PUBLIC_CSV_UPLOAD_URL, {
        version: _constants.API_VERSIONS.public.v1,
        method: 'POST',
        headers: {
          'Content-Type': undefined // Lets the browser set the appropriate content type
        },
        body
      });
    };

    /**
     * List all data source for privilege monitoring engine
     */
    const listPrivMonMonitoredIndices = async ({
      signal
    }) => http.fetch(_constants.MONITORING_ENTITY_LIST_SOURCES_URL, {
      version: _constants.API_VERSIONS.public.v1,
      method: 'GET',
      signal,
      query: {
        type: 'index',
        managed: false,
        name: ENTITY_SOURCE_NAME
      }
    });
    const uploadPrivilegedUserMonitoringFile = async (fileContent, fileName) => {
      const file = new File([new Blob([fileContent])], fileName, {
        type: 'text/csv'
      });
      const body = new FormData();
      body.append('file', file);
      return http.fetch(_constants.MONITORING_USERS_CSV_UPLOAD_URL, {
        version: _constants.API_VERSIONS.public.v1,
        method: 'POST',
        headers: {
          'Content-Type': undefined // Lets the browser set the appropriate content type
        },
        body
      });
    };
    const initPrivilegedMonitoringEngine = () => http.fetch(_constants.MONITORING_ENGINE_INIT_URL, {
      version: _constants.API_VERSIONS.public.v1,
      method: 'POST'
    });
    const fetchPrivilegeMonitoringEngineStatus = () => http.fetch(_constants.PRIVMON_HEALTH_URL, {
      version: _constants.API_VERSIONS.public.v1,
      method: 'GET'
    });
    const fetchPrivilegeMonitoringPrivileges = () => http.fetch(_constants.PRIVMON_PRIVILEGE_CHECK_API, {
      version: _constants.API_VERSIONS.public.v1,
      method: 'GET'
    });

    /**
     * Fetches risk engine settings
     */
    const fetchRiskEngineSettings = () => http.fetch(_constants.RISK_ENGINE_SETTINGS_URL, {
      version: '1',
      method: 'GET'
    });

    /**
     * Deletes Risk engine installation and associated data
     */

    const cleanUpRiskEngine = () => http.fetch(_constants.RISK_ENGINE_CLEANUP_URL, {
      version: '1',
      method: 'DELETE'
    });
    const updateSavedObjectConfiguration = params => http.fetch(_constants.RISK_ENGINE_CONFIGURE_SO_URL, {
      version: _constants.API_VERSIONS.public.v1,
      method: 'PUT',
      body: JSON.stringify(params)
    });
    return {
      fetchRiskScorePreview,
      fetchRiskEngineStatus,
      initRiskEngine,
      enableRiskEngine,
      disableRiskEngine,
      scheduleNowRiskEngine,
      fetchRiskEnginePrivileges,
      fetchAssetCriticalityPrivileges,
      fetchEntityStorePrivileges,
      searchPrivMonIndices,
      createPrivMonImportIndex,
      createAssetCriticality,
      deleteAssetCriticality,
      fetchAssetCriticality,
      fetchAssetCriticalityList,
      uploadAssetCriticalityFile,
      uploadPrivilegedUserMonitoringFile,
      initPrivilegedMonitoringEngine,
      registerPrivMonMonitoredIndices,
      updatePrivMonMonitoredIndices,
      fetchPrivilegeMonitoringEngineStatus,
      fetchPrivilegeMonitoringPrivileges,
      fetchRiskEngineSettings,
      calculateEntityRiskScore,
      cleanUpRiskEngine,
      fetchEntitiesList,
      updateSavedObjectConfiguration,
      listPrivMonMonitoredIndices
    };
  }, [http]);
};
exports.useEntityAnalyticsRoutes = useEntityAnalyticsRoutes;