"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SampleDataManager = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _inferenceCommon = require("@kbn/inference-common");
var _common = require("../../../common");
var _artifact_manager = require("../artifact_manager");
var _index_manager = require("../index_manager");
var _saved_objects_manager = require("../saved_objects_manager");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class SampleDataManager {
  constructor({
    artifactsFolder,
    logger,
    artifactRepositoryUrl,
    elserInferenceId,
    kibanaVersion,
    isServerlessPlatform
  }) {
    (0, _defineProperty2.default)(this, "log", void 0);
    (0, _defineProperty2.default)(this, "artifactManager", void 0);
    (0, _defineProperty2.default)(this, "indexManager", void 0);
    (0, _defineProperty2.default)(this, "savedObjectsManager", void 0);
    (0, _defineProperty2.default)(this, "isInstalling", false);
    this.log = logger;
    this.artifactManager = new _artifact_manager.ArtifactManager({
      artifactsFolder,
      artifactRepositoryUrl,
      kibanaVersion,
      logger: this.log
    });
    this.indexManager = new _index_manager.IndexManager({
      elserInferenceId: elserInferenceId || _inferenceCommon.defaultInferenceEndpoints.ELSER,
      logger: this.log,
      isServerlessPlatform
    });
    this.savedObjectsManager = new _saved_objects_manager.SavedObjectsManager({
      logger: this.log
    });
  }
  async installSampleData({
    sampleType,
    esClient,
    soClient,
    soImporter
  }) {
    this.log.info(`Installing sample data for [${sampleType}]`);
    let archive;
    const indexName = (0, _common.getSampleDataIndexName)(sampleType);
    try {
      const status = await this.getSampleDataStatus({
        sampleType,
        esClient,
        soClient
      });
      if (status.status === 'installed' || this.isInstalling) {
        this.log.warn(`Sample data already installed for [${sampleType}]`);
        return {
          indexName,
          dashboardId: status.dashboardId
        };
      }
      this.isInstalling = true;
      const {
        archive: artifactsArchive,
        manifest,
        mappings
      } = await this.artifactManager.prepareArtifact(sampleType);
      archive = artifactsArchive;
      await this.indexManager.createAndPopulateIndex({
        indexName,
        mappings,
        manifest,
        archive,
        esClient
      });
      this.log.info(`Sample data installation successful for [${sampleType}]`);
      const {
        dashboardId
      } = await this.savedObjectsManager.importSavedObjects(soImporter, sampleType);
      return {
        indexName,
        dashboardId
      };
    } catch (error) {
      await this.indexManager.deleteIndex({
        indexName,
        esClient
      });
      await this.savedObjectsManager.deleteSavedObjects(soClient, sampleType);
      this.log.error(`Sample data installation failed for [${sampleType}]: ${(error === null || error === void 0 ? void 0 : error.message) || error}`);
      throw error;
    } finally {
      try {
        var _archive;
        (_archive = archive) === null || _archive === void 0 ? void 0 : _archive.close();
      } catch (e) {
        this.log.error(`Failed to close archive: ${(e === null || e === void 0 ? void 0 : e.message) || e}`);
      }
      await this.artifactManager.cleanup();
      this.isInstalling = false;
    }
  }
  async removeSampleData({
    sampleType,
    esClient,
    soClient
  }) {
    const indexName = (0, _common.getSampleDataIndexName)(sampleType);
    await this.indexManager.deleteIndex({
      indexName,
      esClient
    });
    if (soClient) {
      await this.savedObjectsManager.deleteSavedObjects(soClient, sampleType);
    }
  }
  async getSampleDataStatus({
    sampleType,
    esClient,
    soClient
  }) {
    const indexName = (0, _common.getSampleDataIndexName)(sampleType);
    try {
      const hasIndex = await this.indexManager.hasIndex({
        indexName,
        esClient
      });
      const dashboardId = await this.savedObjectsManager.getDashboardId(soClient, sampleType);
      if (hasIndex) {
        return {
          status: 'installed',
          indexName,
          dashboardId
        };
      }
      return {
        status: 'uninstalled'
      };
    } catch (error) {
      this.log.warn(`Failed to check sample data status for [${sampleType}]: ${error.message}`);
      return {
        status: 'uninstalled'
      };
    }
  }
}
exports.SampleDataManager = SampleDataManager;