"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getInspectedElementData = void 0;
var _find_first_eui_component = require("./fiber/find_first_eui_component");
var _get_icon_type = require("./dom/get_icon_type");
var _fetch_component_data = require("../api/fetch_component_data");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/** Information about the file where the React component is defined. */

/**
 * Represents information about an inspected component.
 */

/**
 * Options for {@link getInspectedElementData}.
 */

/**
 * Fetches and compiles data about the inspected component.
 * @async
 * @param {GetInspectedElementDataOptions} options
 * @param {HttpStart} options.httpService {@link HttpStart}
 * @param {string | null} options.sourceComponent {@link SourceComponent}
 * @param {ReactFiberNode | null} options.targetFiberNodeWithHtmlElement {@link ReactFiberNode}
 * @returns {Promise<ComponentData | null>} Resolves with {@link ComponentData component data} if found, otherwise null.
 */
const getInspectedElementData = async ({
  httpService,
  sourceComponent,
  targetFiberNode
}) => {
  const debugSource = targetFiberNode === null || targetFiberNode === void 0 ? void 0 : targetFiberNode._debugSource;
  const element = targetFiberNode === null || targetFiberNode === void 0 ? void 0 : targetFiberNode.element;
  if (!debugSource || !element || !sourceComponent) {
    return null;
  }
  const response = await (0, _fetch_component_data.fetchComponentData)({
    httpService,
    fileName: debugSource.fileName
  });
  if (!response) {
    return null;
  }
  const {
    baseFileName,
    codeowners,
    relativePath
  } = response;
  const {
    columnNumber,
    lineNumber,
    fileName
  } = debugSource;
  const fileData = {
    columnNumber,
    lineNumber,
    fileName,
    baseFileName,
    codeowners,
    relativePath
  };
  const iconType = (0, _get_icon_type.getIconType)(element);
  const euiComponentType = (0, _find_first_eui_component.findFirstEuiComponent)(targetFiberNode);
  const euiData = {
    componentType: euiComponentType || 'N/A',
    docsLink: euiComponentType ? `${_constants.EUI_DOCS_BASE}${_constants.EUI_COMPONENTS_DOCS_MAP.get(euiComponentType)}` : `${_constants.EUI_DOCS_BASE}/components`,
    iconType
  };
  const componentData = {
    fileData,
    euiData,
    sourceComponent
  };
  return componentData;
};
exports.getInspectedElementData = getInspectedElementData;