"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getFileEvents = getFileEvents;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const FILE_EVENTS_INDEX_PATTERN = 'logs-endpoint.events.file-*';
const SIZE = 1500;
function getFileEventsQuery({
  endpointIds,
  size,
  gte,
  lte
}) {
  return {
    allow_no_indices: true,
    query: {
      bool: {
        must: [{
          terms: {
            'agent.id': endpointIds
          }
        }, {
          range: {
            '@timestamp': {
              gte: gte !== null && gte !== void 0 ? gte : 'now-24h',
              lte: lte !== null && lte !== void 0 ? lte : 'now'
            }
          }
        }]
      }
    },
    size: 0,
    // Aggregations only
    aggs: {
      unique_process_executable: {
        terms: {
          field: 'process.executable',
          size: size !== null && size !== void 0 ? size : SIZE
        },
        aggs: {
          // Get the latest event for each process.executable
          latest_event: {
            top_hits: {
              size: 1,
              sort: [{
                '@timestamp': {
                  order: 'desc'
                }
              }],
              _source: ['_id', 'agent.id', 'process.executable'] // Include only necessary fields
            }
          }
        }
      }
    },
    ignore_unavailable: true,
    index: [FILE_EVENTS_INDEX_PATTERN]
  };
}
async function getFileEvents(esClient, options) {
  var _result$aggregations$, _result$aggregations;
  const query = getFileEventsQuery(options);
  const result = await esClient.search(query);
  return ((_result$aggregations$ = (_result$aggregations = result.aggregations) === null || _result$aggregations === void 0 ? void 0 : _result$aggregations.unique_process_executable.buckets) !== null && _result$aggregations$ !== void 0 ? _result$aggregations$ : []).map(bucket => {
    const latestEvent = bucket.latest_event.hits.hits[0];
    return {
      _id: [latestEvent._id],
      'agent.id': [latestEvent._source.agent.id],
      'process.executable': [latestEvent._source.process.executable]
    };
  });
}