"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ElasticV3BrowserShipper = void 0;
var tslib_1 = require("tslib");
var rxjs_1 = require("rxjs");
var common_1 = require("../../common");
var common_2 = require("../../common");
/**
 * Elastic V3 shipper to use in the browser.
 */
var ElasticV3BrowserShipper = /** @class */ (function () {
    /**
     * Creates a new instance of the {@link ElasticV3BrowserShipper}.
     * @param options {@link ElasticV3ShipperOptions}
     * @param initContext {@link AnalyticsClientInitContext}
     */
    function ElasticV3BrowserShipper(options, initContext) {
        this.options = options;
        this.initContext = initContext;
        /** Observable to emit the stats of the processed events. */
        this.telemetryCounter$ = new rxjs_1.Subject();
        this.reportTelemetryCounters = (0, common_2.createTelemetryCounterHelper)(this.telemetryCounter$, ElasticV3BrowserShipper.shipperName);
        this.internalQueue$ = new rxjs_1.Subject();
        this.flush$ = new rxjs_1.Subject();
        this.queueFlushed$ = new rxjs_1.Subject();
        this.isOptedIn$ = new rxjs_1.BehaviorSubject(undefined);
        this.clusterUuid = 'UNKNOWN';
        this.setUpInternalQueueSubscriber();
        if (!options.buildShipperHeaders) {
            throw new Error('ElasticV3BrowserShipper requires options.buildShipperHeaders on initialization.');
        }
        if (!options.buildShipperUrl) {
            throw new Error('ElasticV3BrowserShipper requires options.buildShipperUrl on initialization.');
        }
        this.buildHeaders = options.buildShipperHeaders;
        this.url = options.buildShipperUrl({
            channelName: options.channelName,
        });
    }
    /**
     * Uses the `cluster_uuid` and `license_id` from the context to hold them in memory for the generation of the headers
     * used later on in the HTTP request.
     * @param newContext The full new context to set {@link EventContext}
     */
    ElasticV3BrowserShipper.prototype.extendContext = function (newContext) {
        if (newContext.cluster_uuid) {
            this.clusterUuid = newContext.cluster_uuid;
        }
        if (newContext.license_id) {
            this.licenseId = newContext.license_id;
        }
    };
    /**
     * When `false`, it flushes the internal queue and stops sending events.
     * @param isOptedIn `true` for resume sending events. `false` to stop.
     */
    ElasticV3BrowserShipper.prototype.optIn = function (isOptedIn) {
        this.isOptedIn$.next(isOptedIn);
    };
    /**
     * Enqueues the events to be sent to in a batched approach.
     * @param events batched events {@link Event}
     */
    ElasticV3BrowserShipper.prototype.reportEvents = function (events) {
        var _this = this;
        events.forEach(function (event) {
            _this.internalQueue$.next(event);
        });
    };
    /**
     * Triggers a flush of the internal queue to attempt to send any events held in the queue
     * and resolves the returned promise once the queue is emptied.
     */
    ElasticV3BrowserShipper.prototype.flush = function () {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var promise;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        if (this.flush$.isStopped) {
                            // If called after shutdown, return straight away
                            return [2 /*return*/];
                        }
                        promise = (0, rxjs_1.firstValueFrom)(this.queueFlushed$);
                        this.flush$.next();
                        return [4 /*yield*/, promise];
                    case 1:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Shuts down the shipper.
     * Triggers a flush of the internal queue to attempt to send any events held in the queue.
     */
    ElasticV3BrowserShipper.prototype.shutdown = function () {
        this.internalQueue$.complete(); // NOTE: When completing the observable, the buffer logic does not wait and releases any buffered events.
        this.flush$.complete();
    };
    ElasticV3BrowserShipper.prototype.setUpInternalQueueSubscriber = function () {
        var _this = this;
        this.internalQueue$
            .pipe(
        // Buffer events for 1 second or until we have an optIn value
        (0, rxjs_1.bufferWhen)(function () { return (0, rxjs_1.merge)(_this.flush$, (0, rxjs_1.interval)(1000).pipe((0, rxjs_1.skipWhile)(function () { return _this.isOptedIn$.value === undefined; }))); }), 
        // Send events (one batch at a time)
        (0, rxjs_1.concatMap)(function (events) { return tslib_1.__awaiter(_this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        if (!(this.isOptedIn$.value === true && events.length > 0)) return [3 /*break*/, 2];
                        return [4 /*yield*/, this.sendEvents(events)];
                    case 1:
                        _a.sent();
                        _a.label = 2;
                    case 2: return [2 /*return*/];
                }
            });
        }); }), (0, rxjs_1.map)(function () { return _this.queueFlushed$.next(); }))
            .subscribe();
    };
    ElasticV3BrowserShipper.prototype.sendEvents = function (events) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var code, error_1;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        _a.trys.push([0, 2, , 3]);
                        return [4 /*yield*/, this.makeRequest(events)];
                    case 1:
                        code = _a.sent();
                        this.reportTelemetryCounters(events, { code: code });
                        return [3 /*break*/, 3];
                    case 2:
                        error_1 = _a.sent();
                        this.reportTelemetryCounters(events, { code: error_1.code, error: error_1 });
                        return [3 /*break*/, 3];
                    case 3: return [2 /*return*/];
                }
            });
        });
    };
    ElasticV3BrowserShipper.prototype.makeRequest = function (events) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var response, _a, _b, _c, _d, _e, _f, _g;
            return tslib_1.__generator(this, function (_h) {
                switch (_h.label) {
                    case 0: return [4 /*yield*/, fetch(this.url, tslib_1.__assign(tslib_1.__assign({ method: 'POST', body: (0, common_2.eventsToNDJSON)(events), headers: this.buildHeaders(this.clusterUuid, this.options.version, this.licenseId) }, (this.options.debug && { query: { debug: true } })), { 
                            // Allow the request to outlive the page in case the tab is closed
                            keepalive: true }))];
                    case 1:
                        response = _h.sent();
                        if (!this.options.debug) return [3 /*break*/, 3];
                        _b = (_a = this.initContext.logger).debug;
                        _d = (_c = "[".concat(ElasticV3BrowserShipper.shipperName, "]: ").concat(response.status, " - ")).concat;
                        return [4 /*yield*/, response.text()];
                    case 2:
                        _b.apply(_a, [_d.apply(_c, [_h.sent()])]);
                        _h.label = 3;
                    case 3:
                        if (!!response.ok) return [3 /*break*/, 5];
                        _e = common_1.ErrorWithCode.bind;
                        _g = (_f = "".concat(response.status, " - ")).concat;
                        return [4 /*yield*/, response.text()];
                    case 4: throw new (_e.apply(common_1.ErrorWithCode, [void 0, _g.apply(_f, [_h.sent()]), "".concat(response.status)]))();
                    case 5: return [2 /*return*/, "".concat(response.status)];
                }
            });
        });
    };
    /** Shipper's unique name */
    ElasticV3BrowserShipper.shipperName = 'elastic_v3_browser';
    return ElasticV3BrowserShipper;
}());
exports.ElasticV3BrowserShipper = ElasticV3BrowserShipper;
