"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseNDJSON = exports.parseJSONArray = exports.SampleLogsInput = void 0;
var _react = _interopRequireWildcard(require("react"));
var _eui = require("@elastic/eui");
var _fp = require("lodash/fp");
var _public = require("@kbn/kibana-react-plugin/public");
var i18n = _interopRequireWildcard(require("./translations"));
var _state = require("../../state");
var _common = require("../../../../../../common");
var _constants = require("../../../../../../common/constants");
var _jsxFileName = "/opt/buildkite-agent/builds/bk-agent-prod-gcp-1763640253137490323/elastic/kibana-artifacts-snapshot/kibana/x-pack/platform/plugins/shared/automatic_import/public/components/create_integration/create_automatic_import/steps/data_stream_step/sample_logs_input.tsx";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/**
 * Parse the logs sample file content as newiline-delimited JSON (NDJSON).
 *
 * This supports multiline JSON objects if passed multiline flag.
 * Note that in that case the { character must happen at the beginning of the
 * line if and only if it denotes the start of a new JSON object. Thus some
 * inputs that will be parsed as NDJSON without the multiline flag will _not_ be
 * parsed as NDJSON with the multiline flag.
 */
const parseNDJSON = (fileContent, multiline = false) => {
  const separator = multiline ? /\n(?=\{)/ : '\n';
  return fileContent.split(separator) // For multiline, split at newline followed by '{'.
  .filter(entry => entry.trim() !== '') // Remove empty entries.
  .map(entry => JSON.parse(entry)); // Parse each entry as JSON.
};

/**
 * Parse the logs sample file content as a JSON, find an array of entries there.
 *
 * If the JSON object can be parsed, but is not an array, we try to find a candidate
 * among the dictionary keys (it must be identifier-like and its value must be an array).
 *
 * @returns Both the parsed entries and the path to the entries in the JSON object in case of
 * success. Otherwise, an errorNoArrayFound if appropriate. If the parsing failed, raises an error.
 */
exports.parseNDJSON = parseNDJSON;
const parseJSONArray = fileContent => {
  const jsonContent = JSON.parse(fileContent);
  if (Array.isArray(jsonContent)) {
    return {
      entries: jsonContent,
      pathToEntries: [],
      errorNoArrayFound: false
    };
  }
  if (typeof jsonContent === 'object' && jsonContent !== null) {
    const arrayKeys = Object.keys(jsonContent).filter(key => Array.isArray(jsonContent[key]));
    if (arrayKeys.length === 1) {
      const key = arrayKeys[0];
      return {
        entries: jsonContent[key],
        pathToEntries: [key],
        errorNoArrayFound: false
      };
    }
  }
  return {
    errorNoArrayFound: true,
    entries: [],
    pathToEntries: []
  };
};

/**
 * Selects samples from the backend from an array of log samples type T.
 *
 * This is a generic function to apply to arrays of any type.
 *
 * The array is changed in-place so that it will:
 *   - have no more than MaxLogsSampleRows; and
 *   - be shuffled using the reproducible shuffle algorithm;
 *   - however, the first element will be kept in-place.
 *
 * The idea is to perform the same amount of operations on the array
 * regardless of its size and to not use any extra memory.
 *
 * @param array - The array to select from (cannot be empty).
 * @template T - The type of elements in the array.
 * @returns Whether the array was truncated.
 */
exports.parseJSONArray = parseJSONArray;
function trimShuffleLogsSample(array) {
  const willTruncate = array.length > _constants.FRONTEND_SAMPLE_ROWS;
  const numElements = willTruncate ? _constants.FRONTEND_SAMPLE_ROWS : array.length;
  (0, _common.partialShuffleArray)(array, 1, numElements);
  if (willTruncate) {
    array.length = numElements;
  }
  return willTruncate;
}

// The error message structure.

// The parsed logs sample structure.

/**
 * Prepares the logs sample to send to the backend from the user-provided file.
 *
 * This function will return an error message if the file content is not valid, that is:
 *  - it is too large to parse (the memory required is 2-3x of the file size); or
 *  - it looks like a JSON format, but there is no array; or
 *  - it looks like (ND)JSON format, but the items are not JSON dictionaries; or
 *  - the list of entries is empty.
 * In other cases it will parse and return the `logSamples` array of strings.
 *
 * Additionally if the format was (ND)JSON:
 *  - the `samplesFormat` field will be filled out with the format description; and
 *  - the samples will be serialized back to JSON strings;
 * otherwise:
 *  - the `samplesFormat` field will be undefined; and
 *  - the samples will be strings with unknown structure.
 *
 * In all cases it will also:
 *  - shuffle the parsed logs sample using the reproducible shuffle algorithm;
 *  - return no more than MaxLogsSampleRows entries.
 *
 * @param fileContent The content of the user-provided logs sample file.
 * @returns The parsed logs sample structure or an error message.
 */
const prepareLogsContent = fileContent => {
  let parsedJSONContent;
  let jsonSamplesFormat;
  try {
    parsedJSONContent = parseNDJSON(fileContent);

    // Special case for files that can be parsed as both JSON and NDJSON:
    //   for a one-line array [] -> extract its contents (it's a JSON)
    //   for a one-line object {} -> do nothing (keep as NDJSON)
    if (parsedJSONContent.length === 1 && Array.isArray(parsedJSONContent[0])) {
      parsedJSONContent = parsedJSONContent[0];
      jsonSamplesFormat = {
        name: 'json',
        json_path: []
      };
    } else {
      jsonSamplesFormat = {
        name: 'ndjson',
        multiline: false
      };
    }
  } catch (parseNDJSONError) {
    if (parseNDJSONError instanceof RangeError) {
      return {
        error: i18n.LOGS_SAMPLE_ERROR.TOO_LARGE_TO_PARSE
      };
    }
    try {
      const {
        entries,
        pathToEntries,
        errorNoArrayFound
      } = parseJSONArray(fileContent);
      if (errorNoArrayFound) {
        return {
          error: i18n.LOGS_SAMPLE_ERROR.NOT_ARRAY
        };
      }
      parsedJSONContent = entries;
      jsonSamplesFormat = {
        name: 'json',
        json_path: pathToEntries
      };
    } catch (parseJSONError) {
      if (parseJSONError instanceof RangeError) {
        return {
          error: i18n.LOGS_SAMPLE_ERROR.TOO_LARGE_TO_PARSE
        };
      }
      try {
        parsedJSONContent = parseNDJSON(fileContent, true);
        jsonSamplesFormat = {
          name: 'ndjson',
          multiline: true
        };
      } catch (parseMultilineNDJSONError) {
        if (parseMultilineNDJSONError instanceof RangeError) {
          return {
            error: i18n.LOGS_SAMPLE_ERROR.TOO_LARGE_TO_PARSE
          };
        }
        // This is an unknown format, so split into lines and return no samplesFormat.
        const fileLines = fileContent.split('\n').filter(line => line.trim() !== '');
        if (fileLines.length === 0) {
          return {
            error: i18n.LOGS_SAMPLE_ERROR.EMPTY
          };
        }
        const isTruncated = trimShuffleLogsSample(fileLines);
        return {
          samplesFormat: undefined,
          logSamples: fileLines,
          isTruncated
        };
      }
    }
  }

  // This seems to be an ND(JSON), so perform additional checks and return jsonSamplesFormat.

  if (parsedJSONContent.some(log => !(0, _fp.isPlainObject)(log))) {
    return {
      error: i18n.LOGS_SAMPLE_ERROR.NOT_OBJECT
    };
  }
  if (parsedJSONContent.length === 0) {
    return {
      error: i18n.LOGS_SAMPLE_ERROR.EMPTY
    };
  }
  const isTruncated = trimShuffleLogsSample(parsedJSONContent);
  return {
    samplesFormat: jsonSamplesFormat,
    logSamples: parsedJSONContent.map(line => JSON.stringify(line)),
    isTruncated
  };
};
const SampleLogsInput = exports.SampleLogsInput = /*#__PURE__*/_react.default.memo(({
  integrationSettings
}) => {
  const {
    setIntegrationSettings
  } = (0, _state.useActions)();
  const [isParsing, setIsParsing] = (0, _react.useState)(false);
  const [sampleFileError, setSampleFileError] = (0, _react.useState)();
  const {
    docLinks
  } = (0, _public.useKibana)().services;
  const AUTOMATIC_IMPORT_DOCUMENTATION_URL = docLinks === null || docLinks === void 0 ? void 0 : docLinks.links.siem.automaticImport;
  const onChangeLogsSample = (0, _react.useCallback)(files => {
    if (!files) {
      return;
    }
    setSampleFileError(undefined);
    setIntegrationSettings({
      ...integrationSettings,
      logSamples: undefined,
      samplesFormat: undefined
    });
    const logsSampleFile = files[0];
    const reader = new FileReader();
    reader.onloadstart = function () {
      setIsParsing(true);
    };
    reader.onloadend = function () {
      setIsParsing(false);
    };
    reader.onload = function (e) {
      var _e$target;
      const fileContent = (_e$target = e.target) === null || _e$target === void 0 ? void 0 : _e$target.result; // We can safely cast to string since we call `readAsText` to load the file.

      if (fileContent == null) {
        setSampleFileError(i18n.LOGS_SAMPLE_ERROR.CAN_NOT_READ);
        return;
      }
      if (fileContent === '' && e.loaded > 100000) {
        // V8-based browsers can't handle large files and return an empty string
        // instead of an error; see https://stackoverflow.com/a/61316641
        setSampleFileError(i18n.LOGS_SAMPLE_ERROR.TOO_LARGE_TO_PARSE);
        return;
      }
      const prepareResult = prepareLogsContent(fileContent);
      if ('error' in prepareResult) {
        setSampleFileError(prepareResult.error);
        return;
      }
      const {
        samplesFormat,
        logSamples
      } = prepareResult;
      setIntegrationSettings({
        ...integrationSettings,
        logSamples,
        samplesFormat
      });
    };
    const handleReaderError = function () {
      var _reader$error;
      const message = (_reader$error = reader.error) === null || _reader$error === void 0 ? void 0 : _reader$error.message;
      if (message) {
        setSampleFileError(i18n.LOGS_SAMPLE_ERROR.CAN_NOT_READ_WITH_REASON(message));
      } else {
        setSampleFileError(i18n.LOGS_SAMPLE_ERROR.CAN_NOT_READ);
      }
    };
    reader.onerror = handleReaderError;
    reader.onabort = handleReaderError;
    reader.readAsText(logsSampleFile);
  }, [integrationSettings, setIntegrationSettings, setIsParsing]);
  const sampleFileErrorMessage = () => {
    if (!sampleFileError) {
      return null;
    }
    return /*#__PURE__*/_react.default.createElement(_eui.EuiText, {
      color: "danger",
      size: "xs",
      __self: void 0,
      __source: {
        fileName: _jsxFileName,
        lineNumber: 300,
        columnNumber: 7
      }
    }, /*#__PURE__*/_react.default.createElement("span", {
      __self: void 0,
      __source: {
        fileName: _jsxFileName,
        lineNumber: 301,
        columnNumber: 9
      }
    }, `${sampleFileError}. ${i18n.LOGS_SAMPLE_ERROR.HELP_TEXT_PREFIX} `, /*#__PURE__*/_react.default.createElement(_eui.EuiLink, {
      href: AUTOMATIC_IMPORT_DOCUMENTATION_URL,
      target: "_blank",
      external: true,
      __self: void 0,
      __source: {
        fileName: _jsxFileName,
        lineNumber: 303,
        columnNumber: 11
      }
    }, i18n.LOGS_SAMPLE_ERROR.DOCUMENTATION_LINK_TEXT)));
  };
  return /*#__PURE__*/_react.default.createElement(_eui.EuiFormRow, {
    label: /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, i18n.LOGS_SAMPLE_LABEL.MAIN, /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
      size: "s",
      __self: void 0,
      __source: {
        fileName: _jsxFileName,
        lineNumber: 316,
        columnNumber: 11
      }
    }), /*#__PURE__*/_react.default.createElement(_eui.EuiText, {
      color: "subdued",
      size: "xs",
      __self: void 0,
      __source: {
        fileName: _jsxFileName,
        lineNumber: 317,
        columnNumber: 11
      }
    }, `${i18n.LOGS_SAMPLE_LABEL.SUBTEXT_PRETEXT} `, /*#__PURE__*/_react.default.createElement(_eui.EuiLink, {
      href: AUTOMATIC_IMPORT_DOCUMENTATION_URL,
      target: "_blank",
      external: true,
      __self: void 0,
      __source: {
        fileName: _jsxFileName,
        lineNumber: 319,
        columnNumber: 13
      }
    }, i18n.LOGS_SAMPLE_LABEL.SUBTEXT_INFO_LINK))),
    helpText: sampleFileErrorMessage(),
    isInvalid: sampleFileError != null,
    __self: void 0,
    __source: {
      fileName: _jsxFileName,
      lineNumber: 312,
      columnNumber: 5
    }
  }, /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement(_eui.EuiCallOut, {
    iconType: "info",
    color: "warning",
    __self: void 0,
    __source: {
      fileName: _jsxFileName,
      lineNumber: 329,
      columnNumber: 9
    }
  }, i18n.LOGS_SAMPLE_WARNING.MAIN), /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
    size: "s",
    __self: void 0,
    __source: {
      fileName: _jsxFileName,
      lineNumber: 332,
      columnNumber: 9
    }
  }), /*#__PURE__*/_react.default.createElement(_eui.EuiFilePicker, {
    id: "logsSampleFilePicker",
    initialPromptText: /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement(_eui.EuiText, {
      size: "s",
      textAlign: "center",
      __self: void 0,
      __source: {
        fileName: _jsxFileName,
        lineNumber: 338,
        columnNumber: 15
      }
    }, i18n.LOGS_SAMPLE_DESCRIPTION.MAIN), /*#__PURE__*/_react.default.createElement(_eui.EuiText, {
      size: "xs",
      textAlign: "center",
      color: "subdued",
      __self: void 0,
      __source: {
        fileName: _jsxFileName,
        lineNumber: 341,
        columnNumber: 15
      }
    }, i18n.LOGS_SAMPLE_DESCRIPTION.SUBTEXT)),
    onChange: onChangeLogsSample,
    display: "large",
    "aria-label": "Upload logs sample file",
    isLoading: isParsing,
    "data-test-subj": "logsSampleFilePicker",
    "data-loading": isParsing,
    __self: void 0,
    __source: {
      fileName: _jsxFileName,
      lineNumber: 334,
      columnNumber: 9
    }
  })));
});
SampleLogsInput.displayName = 'SampleLogsInput';