"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DEFAULT_TIE_BREAKER_NAME = void 0;
exports.getESQuerySortForTieBreaker = getESQuerySortForTieBreaker;
exports.getESQuerySortForTimeField = getESQuerySortForTimeField;
exports.getEsQuerySort = getEsQuerySort;
exports.getTieBreakerFieldName = getTieBreakerFieldName;
var _constants = require("../../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Returns `EsQuerySort` which is used to sort records in the ES query
 * https://www.elastic.co/guide/en/elasticsearch/reference/current/search-request-sort.html
 * @param sortDir
 * @param timeFieldName
 * @param tieBreakerFieldName
 * @param isTimeNanosBased
 */
function getEsQuerySort({
  sortDir,
  timeFieldName,
  tieBreakerFieldName,
  isTimeNanosBased
}) {
  return [getESQuerySortForTimeField({
    sortDir,
    timeFieldName,
    isTimeNanosBased
  }), getESQuerySortForTieBreaker({
    sortDir,
    tieBreakerFieldName
  })];
}

/**
 * Prepares "sort" structure for a time field for next ES request
 * @param sortDir
 * @param timeFieldName
 * @param isTimeNanosBased
 */
function getESQuerySortForTimeField({
  sortDir,
  timeFieldName,
  isTimeNanosBased
}) {
  return {
    [timeFieldName]: {
      order: sortDir,
      ...(isTimeNanosBased ? {
        format: 'strict_date_optional_time_nanos',
        numeric_type: 'date_nanos'
      } : {
        format: 'strict_date_optional_time'
      })
    }
  };
}

/**
 * Prepares "sort" structure for a tie breaker for next ES request
 * @param sortDir
 * @param tieBreakerFieldName
 */
function getESQuerySortForTieBreaker({
  sortDir,
  tieBreakerFieldName
}) {
  return {
    [tieBreakerFieldName]: sortDir
  };
}

/**
 * The default tie breaker for Discover
 */
const DEFAULT_TIE_BREAKER_NAME = exports.DEFAULT_TIE_BREAKER_NAME = '_doc';

/**
 * The list of field names that are allowed for sorting, but not included in
 * data view fields.
 */
const META_FIELD_NAMES = ['_seq_no', '_doc', '_uid'];

/**
 * Returns a field from the intersection of the set of sortable fields in the
 * given data view and a given set of candidate field names.
 */
function getTieBreakerFieldName(dataView, uiSettings) {
  const sortableFields = (uiSettings.get(_constants.CONTEXT_TIE_BREAKER_FIELDS_SETTING) || []).filter(fieldName => META_FIELD_NAMES.includes(fieldName) || (dataView.fields.getByName(fieldName) || {
    sortable: false
  }).sortable);
  return sortableFields[0];
}