"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RetryService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class RetryService {
  constructor(logger, backOffFactory, serviceName, maxAttempts = 10) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "serviceName", void 0);
    (0, _defineProperty2.default)(this, "maxAttempts", void 0);
    (0, _defineProperty2.default)(this, "backOffStrategy", void 0);
    (0, _defineProperty2.default)(this, "timer", null);
    (0, _defineProperty2.default)(this, "attempt", 0);
    this.logger = logger;
    this.backOffStrategy = backOffFactory.create();
    this.maxAttempts = maxAttempts;
    this.serviceName = serviceName;
  }
  async retryWithBackoff(cb) {
    try {
      this.logger.debug(`[${this.serviceName}][retryWithBackoff] Running. Attempt: ${this.attempt}`, {
        labels: {
          attempt: this.attempt
        }
      });
      const res = await cb();
      this.logger.debug(`[${this.serviceName}][retryWithBackoff] Run successfully after ${this.attempt} attempts.`, {
        labels: {
          attempt: this.attempt
        }
      });
      return res;
    } catch (error) {
      if (this.shouldRetry() && this.isRetryableError(error)) {
        this.stop();
        this.attempt++;
        await this.delay();
        if (error.statusCode) {
          this.logger.warn(`[${this.serviceName}][retryWithBackoff] Failed with status code ${error.statusCode}. Attempt for retry: ${this.attempt}`);
        } else {
          this.logger.warn(`[${this.serviceName}][retryWithBackoff] Failed with error "${error.message}". Attempt for retry: ${this.attempt}`);
        }
        return this.retryWithBackoff(cb);
      }
      throw error;
    } finally {
      this.logger.debug(`[${this.serviceName}][retryWithBackoff] Run ended after ${this.attempt} attempts.`, {
        labels: {
          attempt: this.attempt
        }
      });
    }
  }
  shouldRetry() {
    return this.attempt < this.maxAttempts;
  }
  async delay() {
    const ms = this.backOffStrategy.nextBackOff();
    return new Promise(resolve => {
      this.timer = setTimeout(resolve, ms);
    });
  }
  stop() {
    if (this.timer !== null) {
      clearTimeout(this.timer);
      this.timer = null;
    }
  }
}
exports.RetryService = RetryService;