"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.setupCdrDataViews = exports.installDataView = exports.deleteOldAndLegacyCdrDataViewsForAllSpaces = void 0;
var _common = require("@kbn/spaces-plugin/common");
var _cloudSecurityPostureCommon = require("@kbn/cloud-security-posture-common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DATA_VIEW_TIME_FIELD = '@timestamp';
const getDataViewSafe = async (soClient, currentSpaceId, currentSpaceDataViewId) => {
  try {
    const dataView = await soClient.get('index-pattern', currentSpaceDataViewId, {
      namespace: currentSpaceId
    });
    return dataView;
  } catch (e) {
    return;
  }
};
const getCurrentSpaceId = (spacesService, request) => {
  return (spacesService === null || spacesService === void 0 ? void 0 : spacesService.getSpaceId(request)) || _common.DEFAULT_SPACE_ID;
};
const deleteDataViewSafe = async (soClient, dataViewId, namespace, logger) => {
  try {
    if (namespace === '*') {
      await soClient.delete('index-pattern', dataViewId, {
        force: true
      });
    } else {
      await soClient.delete('index-pattern', dataViewId, {
        namespace
      });
    }
    logger.info(`Deleted old data view: ${dataViewId}`);
  } catch (e) {
    // Ignore if doesn't exist - expected behavior for new installations
    return;
  }
};
const installDataView = async (esClient, soClient, spacesService, dataViewsService, request, dataViewName, indexPattern, dataViewId, logger) => {
  try {
    const currentSpaceId = await getCurrentSpaceId(spacesService, request);
    const currentSpaceDataViewId = `${dataViewId}-${currentSpaceId}`;
    const doesDataViewExist = await getDataViewSafe(soClient, currentSpaceId, currentSpaceDataViewId);
    if (doesDataViewExist) return;
    logger.info(`Creating and saving data view with ID: ${currentSpaceDataViewId}`);
    const dataViewsClient = await dataViewsService.dataViewsServiceFactory(soClient, esClient, request, true);
    await dataViewsClient.createAndSave({
      id: currentSpaceDataViewId,
      title: indexPattern,
      name: `${dataViewName} - ${currentSpaceId} `,
      namespaces: [currentSpaceId],
      allowNoIndex: true,
      timeFieldName: DATA_VIEW_TIME_FIELD
    }, true);
  } catch (error) {
    logger.error(`Failed to setup data view`, error);
  }
};
exports.installDataView = installDataView;
const deleteOldAndLegacyCdrDataViewsForAllSpaces = async (soClient, logger) => {
  try {
    logger.info('Starting deletion of old and legacy CDR data views across all spaces');

    // Get all data views matching old prefixes
    const oldMisconfigurationsPrefixes = _cloudSecurityPostureCommon.CDR_MISCONFIGURATIONS_DATA_VIEW_ID_PREFIX_OLD_VERSIONS;
    const oldVulnerabilitiesPrefixes = _cloudSecurityPostureCommon.CDR_VULNERABILITIES_DATA_VIEW_ID_PREFIX_OLD_VERSIONS;
    const legacyMisconfigurationsPrefixes = _cloudSecurityPostureCommon.CDR_MISCONFIGURATIONS_DATA_VIEW_ID_PREFIX_LEGACY_VERSIONS;
    const legacyVulnerabilitiesPrefixes = _cloudSecurityPostureCommon.CDR_VULNERABILITIES_DATA_VIEW_ID_PREFIX_LEGACY_VERSIONS;
    // Search for all data views across all namespaces and filter by old prefixes
    // We can't use wildcard on _id field, so we fetch all index-patterns and filter in memory
    const allDataViewsResult = await soClient.find({
      type: 'index-pattern',
      namespaces: ['*'],
      // Search across all spaces
      perPage: 1000
    });
    logger.info(`Found ${allDataViewsResult.saved_objects.length} total data views to check`);
    if (allDataViewsResult.total > 1000) {
      logger.warn(`Total data views (${allDataViewsResult.total}) exceeds page limit (1000). Some old data views may not be deleted.`);
    }

    // Filter data views that match old prefixes and legacy ids
    // Include the dash (-) in the check to avoid matching current data views
    const oldMisconfigurationsDataViews = allDataViewsResult.saved_objects.filter(obj => oldMisconfigurationsPrefixes.some(prefix => obj.id.startsWith(`${prefix}-`)));
    const legacyMisconfigurationsDataViews = allDataViewsResult.saved_objects.filter(obj => legacyMisconfigurationsPrefixes.some(prefix => obj.id === prefix));
    const oldVulnerabilitiesDataViews = allDataViewsResult.saved_objects.filter(obj => oldVulnerabilitiesPrefixes.some(prefix => obj.id.startsWith(`${prefix}-`)));
    const legacyVulnerabilitiesDataViews = allDataViewsResult.saved_objects.filter(obj => legacyVulnerabilitiesPrefixes.some(prefix => obj.id === prefix));

    // Delete legacy misconfigurations data views
    for (const dataView of legacyMisconfigurationsDataViews) {
      var _dataView$namespaces;
      const namespace = ((_dataView$namespaces = dataView.namespaces) === null || _dataView$namespaces === void 0 ? void 0 : _dataView$namespaces[0]) || _common.DEFAULT_SPACE_ID;
      logger.info(`Found legacy misconfigurations data view: ${dataView.id} in namespace: ${dataView.namespaces}, deleting...`);
      await deleteDataViewSafe(soClient, dataView.id, namespace, logger);
    }

    // Delete legacy vulnerabilities data views
    for (const dataView of legacyVulnerabilitiesDataViews) {
      var _dataView$namespaces2;
      logger.info(`Found legacy vulnerabilities data view: ${dataView.id}, deleting...`);
      const namespace = ((_dataView$namespaces2 = dataView.namespaces) === null || _dataView$namespaces2 === void 0 ? void 0 : _dataView$namespaces2[0]) || _common.DEFAULT_SPACE_ID;
      await deleteDataViewSafe(soClient, dataView.id, namespace, logger);
    }

    // Delete old misconfigurations data views
    for (const dataView of oldMisconfigurationsDataViews) {
      var _dataView$namespaces3;
      logger.info(`Found old misconfigurations data view: ${dataView.id}, deleting...`);
      const namespace = ((_dataView$namespaces3 = dataView.namespaces) === null || _dataView$namespaces3 === void 0 ? void 0 : _dataView$namespaces3[0]) || _common.DEFAULT_SPACE_ID;
      await deleteDataViewSafe(soClient, dataView.id, namespace, logger);
    }

    // Delete old vulnerabilities data views
    for (const dataView of oldVulnerabilitiesDataViews) {
      var _dataView$namespaces4;
      logger.info(`Found old vulnerabilities data view: ${dataView.id}, deleting...`);
      const namespace = ((_dataView$namespaces4 = dataView.namespaces) === null || _dataView$namespaces4 === void 0 ? void 0 : _dataView$namespaces4[0]) || _common.DEFAULT_SPACE_ID;
      await deleteDataViewSafe(soClient, dataView.id, namespace, logger);
    }
    logger.info('Deletion of old and legacy CDR data views completed successfully');
  } catch (error) {
    logger.error('Failed to delete old and legacy CDR data views', error);
    // Don't throw - deletion failure shouldn't block plugin initialization
  }
};
exports.deleteOldAndLegacyCdrDataViewsForAllSpaces = deleteOldAndLegacyCdrDataViewsForAllSpaces;
const setupCdrDataViews = async (esClient, soClient, spacesService, dataViewsService, request, logger) => {
  await installDataView(esClient, soClient, spacesService, dataViewsService, request, _cloudSecurityPostureCommon.CDR_MISCONFIGURATIONS_DATA_VIEW_NAME, _cloudSecurityPostureCommon.CDR_MISCONFIGURATIONS_INDEX_PATTERN, _cloudSecurityPostureCommon.CDR_MISCONFIGURATIONS_DATA_VIEW_ID_PREFIX, logger);
  await installDataView(esClient, soClient, spacesService, dataViewsService, request, _cloudSecurityPostureCommon.CDR_VULNERABILITIES_DATA_VIEW_NAME, _cloudSecurityPostureCommon.CDR_VULNERABILITIES_INDEX_PATTERN, _cloudSecurityPostureCommon.CDR_VULNERABILITIES_DATA_VIEW_ID_PREFIX, logger);
};
exports.setupCdrDataViews = setupCdrDataViews;