"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.groupingHandlerFactory = void 0;
var _lodash = require("lodash");
var _async = require("async");
var _i18n = require("@kbn/i18n");
var _queue_field_candidates = require("@kbn/aiops-log-rate-analysis/queue_field_candidates");
var _stream_reducer = require("@kbn/aiops-log-rate-analysis/api/stream_reducer");
var _is_request_aborted_error = require("@kbn/aiops-common/is_request_aborted_error");
var _fetch_frequent_item_sets = require("@kbn/aiops-log-rate-analysis/queries/fetch_frequent_item_sets");
var _fetch_terms_2_categories_counts = require("@kbn/aiops-log-rate-analysis/queries/fetch_terms_2_categories_counts");
var _get_significant_item_groups = require("@kbn/aiops-log-rate-analysis/queries/get_significant_item_groups");
var _fetch_mini_histograms_for_significant_groups = require("@kbn/aiops-log-rate-analysis/queries/fetch_mini_histograms_for_significant_groups");
var _constants = require("../response_stream_utils/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const groupingHandlerFactory = ({
  abortSignal,
  esClient,
  requestBody,
  responseStream,
  logDebugMessage,
  logger,
  stateHandler
}) => async (significantCategories, significantTerms, overallTimeSeries) => {
  logDebugMessage('Group results.');
  function pushHistogramDataLoadingState() {
    responseStream.push((0, _stream_reducer.updateLoadingState)({
      ccsWarning: false,
      loaded: stateHandler.loaded(),
      loadingState: _i18n.i18n.translate('xpack.aiops.logRateAnalysis.loadingState.loadingHistogramData', {
        defaultMessage: 'Loading histogram data.'
      })
    }));
  }
  responseStream.push((0, _stream_reducer.updateLoadingState)({
    ccsWarning: false,
    loaded: stateHandler.loaded(),
    loadingState: _i18n.i18n.translate('xpack.aiops.logRateAnalysis.loadingState.groupingResults', {
      defaultMessage: 'Transforming significant field/value pairs into groups.'
    }),
    groupsMissing: true
  }));
  try {
    const {
      fields,
      itemSets
    } = await (0, _fetch_frequent_item_sets.fetchFrequentItemSets)({
      esClient,
      logger,
      emitError: responseStream.pushError,
      abortSignal,
      arguments: {
        index: requestBody.index,
        searchQuery: JSON.parse(requestBody.searchQuery),
        significantItems: significantTerms,
        timeFieldName: requestBody.timeFieldName,
        deviationMin: requestBody.deviationMin,
        deviationMax: requestBody.deviationMax,
        sampleProbability: stateHandler.sampleProbability()
      }
    });
    if (significantCategories.length > 0 && significantTerms.length > 0) {
      const {
        fields: significantCategoriesFields,
        itemSets: significantCategoriesItemSets
      } = await (0, _fetch_terms_2_categories_counts.fetchTerms2CategoriesCounts)(esClient, requestBody, JSON.parse(requestBody.searchQuery), significantTerms, itemSets, significantCategories, requestBody.deviationMin, requestBody.deviationMax, logger, responseStream.pushError, abortSignal);
      fields.push(...significantCategoriesFields);
      itemSets.push(...significantCategoriesItemSets);
    }
    if (stateHandler.shouldStop()) {
      logDebugMessage('shouldStop after fetching frequent_item_sets.');
      responseStream.end();
      return;
    }
    if (fields.length > 0 && itemSets.length > 0) {
      const significantItemGroups = (0, _get_significant_item_groups.getSignificantItemGroups)(itemSets, [...significantTerms, ...significantCategories], fields);

      // We'll find out if there's at least one group with at least two items,
      // only then will we return the groups to the clients and make the grouping option available.
      const maxItems = Math.max(...significantItemGroups.map(g => g.group.length));
      if (maxItems > 1) {
        responseStream.push((0, _stream_reducer.addSignificantItemsGroup)(significantItemGroups));
      }
      stateHandler.loaded(_constants.LOADED_FIELD_CANDIDATES + _constants.PROGRESS_STEP_P_VALUES + _constants.PROGRESS_STEP_HISTOGRAMS + _constants.PROGRESS_STEP_GROUPING);
      pushHistogramDataLoadingState();
      if (stateHandler.shouldStop()) {
        logDebugMessage('shouldStop after grouping.');
        responseStream.end();
        return;
      }
      logDebugMessage(`Fetch ${significantItemGroups.length} group histograms.`);
      const groupHistogramQueueChunks = (0, _lodash.chunk)(significantItemGroups, _queue_field_candidates.QUEUE_CHUNKING_SIZE);
      const loadingStepSize = 1 / groupHistogramQueueChunks.length * _constants.PROGRESS_STEP_HISTOGRAMS_GROUPS;
      const groupHistogramQueue = (0, _async.queue)(async function (payload) {
        if (stateHandler.shouldStop()) {
          logDebugMessage('shouldStop abort fetching group histograms.');
          groupHistogramQueue.kill();
          responseStream.end();
          return;
        }
        if (overallTimeSeries !== undefined) {
          let histograms;
          try {
            histograms = await (0, _fetch_mini_histograms_for_significant_groups.fetchMiniHistogramsForSignificantGroups)(esClient, requestBody, payload, overallTimeSeries, logger, stateHandler.sampleProbability(), () => {}, abortSignal);
          } catch (e) {
            logger.error(`Failed to fetch the histogram data chunk for groups, got: \n${e.toString()}`);
            responseStream.pushError(`Failed to fetch the histogram data chunk for groups.`);
            return;
          }
          stateHandler.loaded(loadingStepSize, false);
          pushHistogramDataLoadingState();
          responseStream.push((0, _stream_reducer.addSignificantItemsGroupHistogram)(histograms));
        }
      }, _constants.MAX_CONCURRENT_QUERIES);
      await groupHistogramQueue.push(groupHistogramQueueChunks);
      await groupHistogramQueue.drain();
    }
  } catch (e) {
    if (!(0, _is_request_aborted_error.isRequestAbortedError)(e)) {
      logger.error(`Failed to transform field/value pairs into groups, got: \n${e.toString()}`);
      responseStream.pushError(`Failed to transform field/value pairs into groups.`);
    }
  }
};
exports.groupingHandlerFactory = groupingHandlerFactory;