/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.inference.logging;

import java.io.Closeable;
import java.time.Clock;
import java.time.Instant;
import java.util.Iterator;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogBuilder;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.common.Strings;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.threadpool.Scheduler;
import org.elasticsearch.threadpool.ThreadPool;

public class Throttler
implements Closeable {
    private static final Logger classLogger = LogManager.getLogger(Throttler.class);
    private final TimeValue loggingInterval;
    private final Clock clock;
    private final ConcurrentMap<String, LogExecutor> logExecutors;
    private final AtomicReference<Scheduler.Cancellable> cancellableTask = new AtomicReference();
    private final AtomicBoolean isRunning = new AtomicBoolean(true);
    private final ThreadPool threadPool;
    private final ReentrantReadWriteLock lock = new ReentrantReadWriteLock();

    public Throttler(TimeValue loggingInterval, ThreadPool threadPool) {
        this(loggingInterval, Clock.systemUTC(), threadPool, new ConcurrentHashMap<String, LogExecutor>());
    }

    public Throttler(Throttler oldThrottler, TimeValue loggingInterval) {
        this(loggingInterval, oldThrottler.clock, oldThrottler.threadPool, new ConcurrentHashMap<String, LogExecutor>(oldThrottler.logExecutors));
    }

    Throttler(TimeValue loggingInterval, Clock clock, ThreadPool threadPool, ConcurrentMap<String, LogExecutor> logExecutors) {
        this.threadPool = Objects.requireNonNull(threadPool);
        this.loggingInterval = Objects.requireNonNull(loggingInterval);
        this.clock = Objects.requireNonNull(clock);
        this.logExecutors = Objects.requireNonNull(logExecutors);
    }

    public void init() {
        this.cancellableTask.set(this.startRepeatingLogEmitter());
    }

    private Scheduler.Cancellable startRepeatingLogEmitter() {
        classLogger.debug(() -> Strings.format((String)"Scheduling repeating log emitter with interval [%s]", (Object[])new Object[]{this.loggingInterval}));
        return this.threadPool.scheduleWithFixedDelay(this::emitRepeatedLogs, this.loggingInterval, (Executor)this.threadPool.executor("inference_utility"));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void emitRepeatedLogs() {
        if (!this.isRunning.get()) {
            return;
        }
        ReentrantReadWriteLock.WriteLock writeLock = this.lock.writeLock();
        writeLock.lock();
        try {
            Iterator iter = this.logExecutors.values().iterator();
            while (iter.hasNext()) {
                LogExecutor executor = (LogExecutor)iter.next();
                executor.logRepeatedMessages();
                iter.remove();
            }
        }
        finally {
            writeLock.unlock();
        }
    }

    public void execute(Logger logger, Level level, String message, Throwable t) {
        this.executeInternal(logger, level, message, t);
    }

    public void execute(Logger logger, Level level, String message) {
        this.executeInternal(logger, level, message, null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void executeInternal(Logger logger, Level level, String message, Throwable throwable) {
        if (!this.isRunning.get()) {
            return;
        }
        ReentrantReadWriteLock.ReadLock readLock = this.lock.readLock();
        readLock.lock();
        try {
            LogExecutor logExecutor = this.logExecutors.compute(message, (key, value) -> Objects.requireNonNullElseGet(value, () -> new LogExecutor(this.clock, logger, level, message, throwable)));
            logExecutor.logFirstMessage();
        }
        finally {
            readLock.unlock();
        }
    }

    @Override
    public void close() {
        this.isRunning.set(false);
        if (this.cancellableTask.get() != null) {
            this.cancellableTask.get().cancel();
        }
        this.clearLogExecutors();
    }

    private void clearLogExecutors() {
        ReentrantReadWriteLock.WriteLock writeLock = this.lock.writeLock();
        writeLock.lock();
        try {
            this.logExecutors.clear();
        }
        finally {
            writeLock.unlock();
        }
    }

    private static class LogExecutor {
        private static final long INITIAL_LOG_COUNTER_VALUE = -1L;
        private final AtomicLong skippedLogCalls = new AtomicLong(-1L);
        private final AtomicReference<Instant> timeOfLastLogCall;
        private final Clock clock;
        private final Logger throttledLogger;
        private final Level level;
        private final String originalMessage;
        private final Throwable throwable;

        LogExecutor(Clock clock, Logger logger, Level level, String originalMessage, @Nullable Throwable throwable) {
            this.clock = Objects.requireNonNull(clock);
            this.timeOfLastLogCall = new AtomicReference<Instant>(Instant.now(this.clock));
            this.throttledLogger = Objects.requireNonNull(logger);
            this.level = Objects.requireNonNull(level);
            this.originalMessage = Objects.requireNonNull(originalMessage);
            this.throwable = throwable;
        }

        void logRepeatedMessages() {
            long numSkippedLogCalls = this.skippedLogCalls.get();
            if (!LogExecutor.hasRepeatedLogsToEmit(numSkippedLogCalls)) {
                return;
            }
            String enrichedMessage = numSkippedLogCalls == 1L ? Strings.format((String)"%s, repeated 1 time, last message at [%s]", (Object[])new Object[]{this.originalMessage, this.timeOfLastLogCall.get()}) : Strings.format((String)"%s, repeated %s times, last message at [%s]", (Object[])new Object[]{this.originalMessage, this.skippedLogCalls, this.timeOfLastLogCall.get()});
            this.log(enrichedMessage);
        }

        private void log(String enrichedMessage) {
            LogBuilder builder = this.throttledLogger.atLevel(this.level);
            if (this.throwable != null) {
                builder = builder.withThrowable(this.throwable);
            }
            builder.log(enrichedMessage);
        }

        private static boolean hasRepeatedLogsToEmit(long numSkippedLogCalls) {
            return numSkippedLogCalls > 0L;
        }

        void logFirstMessage() {
            this.timeOfLastLogCall.set(Instant.now(this.clock));
            if (!LogExecutor.hasLoggedOriginalMessage(this.skippedLogCalls.getAndIncrement())) {
                this.log(this.originalMessage);
            }
        }

        private static boolean hasLoggedOriginalMessage(long numSkippedLogCalls) {
            return numSkippedLogCalls >= 0L;
        }
    }
}

