"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createEmptyLensState = createEmptyLensState;
exports.deserializeState = deserializeState;
exports.extractInheritedViewModeObservable = extractInheritedViewModeObservable;
exports.getParentContext = getParentContext;
exports.getRenderMode = getRenderMode;
exports.getStructuredDatasourceStates = getStructuredDatasourceStates;
exports.getViewMode = getViewMode;
exports.isTextBasedLanguage = isTextBasedLanguage;
var _presentationPublishing = require("@kbn/presentation-publishing");
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _esQuery = require("@kbn/es-query");
var _esql = require("./esql");
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function createEmptyLensState(visualizationType = null, title, description, query, filters) {
  const isTextBased = query && (0, _esQuery.isOfAggregateQueryType)(query);
  return {
    attributes: {
      version: _constants.LENS_ITEM_LATEST_VERSION,
      title: title !== null && title !== void 0 ? title : '',
      description: description !== null && description !== void 0 ? description : '',
      visualizationType,
      references: [],
      state: {
        query: query || {
          query: '',
          language: 'kuery'
        },
        filters: filters || [],
        internalReferences: [],
        datasourceStates: {
          ...(isTextBased ? {
            textBased: {}
          } : {
            formBased: {}
          })
        },
        visualization: {}
      }
    }
  };
}

/**
 * Shared logic to ensure the attributes are correctly loaded
 * Make sure to inject references from the container down to the runtime state
 * this ensure migrations/copy to spaces works correctly
 **/
async function deserializeState({
  attributeService,
  ...services
}, rawState, references) {
  var _savedObjectRef$id;
  const fallbackAttributes = createEmptyLensState().attributes;
  const savedObjectRef = (0, _presentationPublishing.findSavedObjectRef)(_constants.DOC_TYPE, references);
  const savedObjectId = (_savedObjectRef$id = savedObjectRef === null || savedObjectRef === void 0 ? void 0 : savedObjectRef.id) !== null && _savedObjectRef$id !== void 0 ? _savedObjectRef$id : rawState.savedObjectId;
  if (savedObjectId) {
    try {
      const {
        attributes,
        managed,
        sharingSavedObjectProps
      } = await attributeService.loadFromLibrary(savedObjectId);
      return {
        ...rawState,
        savedObjectId,
        attributes,
        managed,
        sharingSavedObjectProps
      };
    } catch (e) {
      // return an empty Lens document if no saved object is found
      return {
        ...rawState,
        attributes: fallbackAttributes
      };
    }
  }

  // Inject applied only to by-value SOs
  const newState = attributeService.injectReferences('attributes' in rawState ? rawState : {
    attributes: rawState
  }, references !== null && references !== void 0 && references.length ? references : undefined);
  if (newState.isNewPanel) {
    try {
      var _ref;
      const newAttributes = await (0, _esql.loadESQLAttributes)(services);
      // provide a fallback
      return {
        ...newState,
        attributes: (_ref = newAttributes !== null && newAttributes !== void 0 ? newAttributes : newState.attributes) !== null && _ref !== void 0 ? _ref : fallbackAttributes
      };
    } catch (e) {
      // return an empty Lens document if no saved object is found
      return {
        ...newState,
        attributes: fallbackAttributes
      };
    }
  }
  return newState;
}
function isTextBasedLanguage(state) {
  var _state$attributes;
  return (0, _esQuery.isOfAggregateQueryType)((_state$attributes = state.attributes) === null || _state$attributes === void 0 ? void 0 : _state$attributes.state.query);
}
function getViewMode(api) {
  return (0, _presentationPublishing.apiPublishesViewMode)(api) ? (0, _presentationPublishing.getInheritedViewMode)(api) : undefined;
}
function getRenderMode(api) {
  var _getViewMode;
  const mode = (_getViewMode = getViewMode(api)) !== null && _getViewMode !== void 0 ? _getViewMode : 'view';
  return mode === 'print' ? 'view' : mode;
}
function apiHasExecutionContextFunction(api) {
  return (0, _lodash.isObject)(api) && 'getAppContext' in api && typeof api.getAppContext === 'function';
}
function getParentContext(parentApi) {
  if ((0, _presentationPublishing.apiHasExecutionContext)(parentApi)) {
    return parentApi.executionContext;
  }
  if (apiHasExecutionContextFunction(parentApi)) {
    return {
      type: parentApi.getAppContext().currentAppId
    };
  }
  return;
}
function extractInheritedViewModeObservable(parentApi) {
  if ((0, _presentationPublishing.apiPublishesViewMode)(parentApi)) {
    return parentApi.viewMode$;
  }
  if ((0, _presentationPublishing.apiHasParentApi)(parentApi)) {
    return extractInheritedViewModeObservable(parentApi.parentApi);
  }
  return new _rxjs.BehaviorSubject('view');
}
function getStructuredDatasourceStates(datasourceStates) {
  var _ref2, _formBased$state, _formBased, _ref3, _textBased$state, _textBased;
  return {
    formBased: (_ref2 = (_formBased$state = datasourceStates === null || datasourceStates === void 0 ? void 0 : (_formBased = datasourceStates.formBased) === null || _formBased === void 0 ? void 0 : _formBased.state) !== null && _formBased$state !== void 0 ? _formBased$state : datasourceStates === null || datasourceStates === void 0 ? void 0 : datasourceStates.formBased) !== null && _ref2 !== void 0 ? _ref2 : undefined,
    textBased: (_ref3 = (_textBased$state = datasourceStates === null || datasourceStates === void 0 ? void 0 : (_textBased = datasourceStates.textBased) === null || _textBased === void 0 ? void 0 : _textBased.state) !== null && _textBased$state !== void 0 ? _textBased$state : datasourceStates === null || datasourceStates === void 0 ? void 0 : datasourceStates.textBased) !== null && _ref3 !== void 0 ? _ref3 : undefined
  };
}