"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.REFRESH_CONFIG_INTERVAL_MS = exports.ConfigurationService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _artifact = require("./artifact.errors");
var _indices_metadata = require("./indices_metadata.types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const REFRESH_CONFIG_INTERVAL_MS = exports.REFRESH_CONFIG_INTERVAL_MS = 60 * 60 * 1000;
const CONFIGURATION_ARTIFACT_NAME = 'indices-metadata-configuration-v1';
class ConfigurationService {
  constructor(logger) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "artifactService", void 0);
    (0, _defineProperty2.default)(this, "indicesMetadataConfiguration$", void 0);
    (0, _defineProperty2.default)(this, "stop$", new _rxjs.ReplaySubject(1));
    this.logger = logger.get(ConfigurationService.name);
  }
  start(artifactService, defaultConfiguration) {
    this.artifactService = artifactService;
    this.indicesMetadataConfiguration$ = (0, _rxjs.timer)(0, REFRESH_CONFIG_INTERVAL_MS).pipe((0, _rxjs.exhaustMap)(() => this.getConfiguration()), (0, _rxjs.takeUntil)(this.stop$), (0, _rxjs.startWith)(defaultConfiguration), (0, _rxjs.filter)(config => config !== undefined), (0, _rxjs.distinctUntilChanged)(), (0, _rxjs.shareReplay)(1));
  }
  stop() {
    this.stop$.next();
    this.stop$.complete();
  }
  getIndicesMetadataConfiguration$() {
    this.ensureStarted();
    return this.indicesMetadataConfiguration$;
  }
  async getConfiguration() {
    this.ensureStarted();
    try {
      this.logger.debug('Getting indices metadata configuration');
      const artifact = await this.artifactService.getArtifact(CONFIGURATION_ARTIFACT_NAME);
      if (!artifact.modified) {
        this.logger.debug('Indices metadata configuration has not been modified');
        return undefined;
      }
      this.logger.debug('Indices metadata configuration has been modified', {
        artifact
      });
      return _indices_metadata.IndicesMetadataConfigurationSchema.validate(artifact.data);
    } catch (error) {
      if (error instanceof _artifact.ManifestNotFoundError) {
        this.logger.warn('Indices metadata configuration manifest not found');
      } else if (error instanceof _artifact.ArtifactNotFoundError) {
        this.logger.warn('Indices metadata configuration artifact not found');
      } else {
        this.logger.error(`Failed to get indices metadata configuration: ${error}`, {
          error
        });
      }
      return undefined;
    }
  }
  ensureStarted() {
    if (!this.artifactService || !this.indicesMetadataConfiguration$) {
      throw new Error('Configuration service not started');
    }
  }
}
exports.ConfigurationService = ConfigurationService;