"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getRollingUpgradeOptions = exports.UpgradeActionRunner = exports.MINIMUM_EXECUTION_DURATION_SECONDS = exports.EXPIRATION_DURATION_SECONDS = void 0;
exports.upgradeBatch = upgradeBatch;
var _uuid = require("uuid");
var _moment = _interopRequireDefault(require("moment"));
var _gte = _interopRequireDefault(require("semver/functions/gte"));
var _services = require("../../../common/services");
var _errors = require("../../errors");
var _app_context = require("../app_context");
var _action_runner = require("./action_runner");
var _crud = require("./crud");
var _actions = require("./actions");
var _hosted_agent = require("./hosted_agent");
var _bulk_action_types = require("./bulk_action_types");
var _action_status = require("./action_status");
var _versions = require("./versions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class UpgradeActionRunner extends _action_runner.ActionRunner {
  async processAgents(agents) {
    var _this$actionParams, _this$actionParams2;
    return await upgradeBatch(this.esClient, agents, {}, this.actionParams, (_this$actionParams = this.actionParams) !== null && _this$actionParams !== void 0 && _this$actionParams.spaceId ? [(_this$actionParams2 = this.actionParams) === null || _this$actionParams2 === void 0 ? void 0 : _this$actionParams2.spaceId] : undefined);
  }
  getTaskType() {
    return _bulk_action_types.BulkActionTaskType.UPGRADE_RETRY;
  }
  getActionType() {
    return 'UPGRADE';
  }
}
exports.UpgradeActionRunner = UpgradeActionRunner;
const isActionIdCancelled = async (esClient, actionId) => {
  const cancelledActions = await (0, _action_status.getCancelledActions)(esClient);
  return cancelledActions.filter(action => action.actionId === actionId).length > 0;
};
async function upgradeBatch(esClient, givenAgents, outgoingErrors, options, spaceIds) {
  var _options$actionId, _options$total, _agentsToUpdate$;
  const soClient = _app_context.appContextService.getInternalUserSOClientForSpaceId(spaceIds === null || spaceIds === void 0 ? void 0 : spaceIds[0]);
  const errors = {
    ...outgoingErrors
  };
  const hostedPolicies = await (0, _hosted_agent.getHostedPolicies)(soClient, givenAgents);

  // results from getAgents with options.kuery '' (or even 'active:false') may include hosted agents
  // filter them out unless options.force
  const agentsToCheckUpgradeable = 'kuery' in options && !options.force ? givenAgents.filter(agent => !(0, _hosted_agent.isHostedAgent)(hostedPolicies, agent)) : givenAgents;
  const latestAgentVersion = await (0, _versions.getLatestAvailableAgentVersion)();
  const upgradeableResults = await Promise.allSettled(agentsToCheckUpgradeable.map(async agent => {
    // Filter out agents that are:
    //  - currently unenrolling
    //  - unenrolled
    //  - recently upgraded
    //  - currently upgrading
    //  - upgradeable b/c of version check
    const isNotAllowed = !options.skipRateLimitCheck && (0, _services.getRecentUpgradeInfoForAgent)(agent).hasBeenUpgradedRecently || !options.force && !options.skipRateLimitCheck && !(0, _services.isAgentUpgradeableToVersion)(agent, options.version);
    if (isNotAllowed) {
      throw new _errors.FleetError(`Agent ${agent.id} is not upgradeable: ${(0, _services.getNotUpgradeableMessage)(agent, latestAgentVersion, options.version)}`);
    }
    if (!options.force && (0, _hosted_agent.isHostedAgent)(hostedPolicies, agent)) {
      throw new _errors.HostedAgentPolicyRestrictionRelatedError(`Cannot upgrade agent in hosted agent policy ${agent.policy_id}`);
    }
    return agent;
  }));

  // Filter & record errors from results
  const agentsToUpdate = upgradeableResults.reduce((agents, result, index) => {
    if (result.status === 'fulfilled') {
      agents.push(result.value);
    } else {
      const id = givenAgents[index].id;
      errors[id] = result.reason;
    }
    return agents;
  }, []);

  // Create upgrade action for each agent
  const now = new Date().toISOString();
  const data = {
    version: options.version,
    sourceURI: options.sourceUri
  };
  const rollingUpgradeOptions = getRollingUpgradeOptions(options === null || options === void 0 ? void 0 : options.startTime, options.upgradeDurationSeconds);
  if (options.actionId && (await isActionIdCancelled(esClient, options.actionId))) {
    _app_context.appContextService.getLogger().info(`Skipping batch of actionId:${options.actionId} of ${givenAgents.length} agents as the upgrade was cancelled`);
    return {
      actionId: options.actionId
    };
  }
  if (options.actionId) {
    _app_context.appContextService.getLogger().info(`Continuing batch of actionId:${options.actionId} of ${givenAgents.length} agents of upgrade`);
  }
  await (0, _crud.bulkUpdateAgents)(esClient, agentsToUpdate.map(agent => {
    var _agent$agent$version, _agent$agent, _agent$upgrade_attemp;
    return {
      agentId: agent.id,
      data: {
        upgraded_at: null,
        upgrade_started_at: now,
        ...(options.isAutomatic && (0, _gte.default)((_agent$agent$version = (_agent$agent = agent.agent) === null || _agent$agent === void 0 ? void 0 : _agent$agent.version) !== null && _agent$agent$version !== void 0 ? _agent$agent$version : '0.0.0', _services.AGENT_UPGARDE_DETAILS_SUPPORTED_VERSION) ? {
          upgrade_attempts: [now, ...((_agent$upgrade_attemp = agent.upgrade_attempts) !== null && _agent$upgrade_attemp !== void 0 ? _agent$upgrade_attemp : [])]
        } : {})
      }
    };
  }), errors);
  const actionId = (_options$actionId = options.actionId) !== null && _options$actionId !== void 0 ? _options$actionId : (0, _uuid.v4)();
  const total = (_options$total = options.total) !== null && _options$total !== void 0 ? _options$total : givenAgents.length;
  const namespaces = spaceIds ? spaceIds : [];
  await (0, _actions.createAgentAction)(esClient, soClient, {
    id: actionId,
    created_at: now,
    data,
    ack_data: data,
    type: 'UPGRADE',
    total,
    agents: agentsToUpdate.map(agent => agent.id),
    ...rollingUpgradeOptions,
    namespaces,
    is_automatic: options.isAutomatic,
    policyId: (_agentsToUpdate$ = agentsToUpdate[0]) === null || _agentsToUpdate$ === void 0 ? void 0 : _agentsToUpdate$.policy_id
  });
  await (0, _actions.createErrorActionResults)(esClient, actionId, errors, 'cannot upgrade hosted agent or agent not upgradeable');
  return {
    actionId
  };
}
const MINIMUM_EXECUTION_DURATION_SECONDS = exports.MINIMUM_EXECUTION_DURATION_SECONDS = 60 * 60 * 2; // 2h
const EXPIRATION_DURATION_SECONDS = exports.EXPIRATION_DURATION_SECONDS = 60 * 60 * 24 * 30; // 1 month

const getRollingUpgradeOptions = (startTime, upgradeDurationSeconds) => {
  const now = new Date().toISOString();
  // Expiration time is set to a very long value (1 month) to allow upgrading agents staying offline for long time
  const expiration = (0, _moment.default)(startTime).add(EXPIRATION_DURATION_SECONDS, 'seconds').toISOString();
  // Perform a rolling upgrade
  if (upgradeDurationSeconds) {
    const minExecutionDuration = Math.min(MINIMUM_EXECUTION_DURATION_SECONDS, upgradeDurationSeconds);
    return {
      start_time: startTime !== null && startTime !== void 0 ? startTime : now,
      rollout_duration_seconds: upgradeDurationSeconds,
      minimum_execution_duration: minExecutionDuration,
      expiration
    };
  }
  // Schedule without rolling upgrade (Immediately after start_time)
  if (startTime && !upgradeDurationSeconds) {
    return {
      start_time: startTime !== null && startTime !== void 0 ? startTime : now,
      minimum_execution_duration: MINIMUM_EXECUTION_DURATION_SECONDS,
      expiration
    };
  } else {
    // Regular bulk upgrade (non scheduled, non rolling)
    return {};
  }
};
exports.getRollingUpgradeOptions = getRollingUpgradeOptions;