"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useMultiContent = useMultiContent;
var _react = require("react");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function useMultiContent({
  defaultValue,
  onChange
}) {
  /**
   * Each content validity is kept in this state. When updating a content with "updateContentAt()", we
   * update the state validity and trigger a re-render.
   */
  const [validation, setValidation] = (0, _react.useState)({
    isValid: true,
    contents: {}
  });

  /**
   * The updated data where a content current data is merged when it unmounts
   */
  const [stateData, setStateData] = (0, _react.useState)(defaultValue !== null && defaultValue !== void 0 ? defaultValue : {});

  /**
   * A map object of all the active content(s) present in the DOM. In a multi step
   * form wizard, there is only 1 content at the time in the DOM, but in long vertical
   * flow content, multiple content could be present.
   * When a content unmounts it will remove itself from this map.
   */
  const contents = (0, _react.useRef)({});
  const updateContentDataAt = (0, _react.useCallback)(function (updatedData) {
    setStateData(prev => ({
      ...prev,
      ...updatedData
    }));
  }, []);

  /**
   * Read the multi content data.
   */
  const getData = (0, _react.useCallback)(() => {
    /**
     * If there is one or more active content(s) in the DOM, and it is valid,
     * we read its data and merge it into our stateData before returning it.
     */
    const activeContentData = {};
    for (const [id, _content] of Object.entries(contents.current)) {
      if (validation.contents[id] !== false) {
        const contentData = _content.getData();

        // Replace the getData() handler with the cached value
        _content.getData = () => contentData;
        activeContentData[id] = contentData;
      }
    }
    return {
      ...stateData,
      ...activeContentData
    };
  }, [stateData, validation]);

  /**
   * Read a single content data.
   */
  const getSingleContentData = (0, _react.useCallback)(contentId => {
    if (contents.current[contentId]) {
      return contents.current[contentId].getData();
    }
    return stateData[contentId];
  }, [stateData]);
  const updateContentValidity = (0, _react.useCallback)(updatedData => {
    let isAllContentValid = validation.isValid;
    setValidation(prev => {
      if (Object.entries(updatedData).every(([contentId, isValid]) => prev.contents[contentId] === isValid)) {
        // No change in validation, nothing to update
        isAllContentValid = prev.isValid;
        return prev;
      }
      const nextContentsValidityState = {
        ...prev.contents,
        ...updatedData
      };
      isAllContentValid = Object.values(nextContentsValidityState).some(_isValid => _isValid === undefined) ? undefined : Object.values(nextContentsValidityState).every(Boolean);
      return {
        isValid: isAllContentValid,
        contents: nextContentsValidityState
      };
    });
    return isAllContentValid;
  }, [validation.isValid]);

  /**
   * Validate the content(s) currently in the DOM
   */
  const validate = (0, _react.useCallback)(async () => {
    if (Object.keys(contents.current).length === 0) {
      return Boolean(validation.isValid);
    }
    const updatedValidation = {};
    for (const [id, _content] of Object.entries(contents.current)) {
      const isValid = await _content.validate();
      _content.validate = async () => isValid;
      updatedValidation[id] = isValid;
    }
    return Boolean(updateContentValidity(updatedValidation));
  }, [validation.isValid, updateContentValidity]);

  /**
   * Update a content. It replaces the content in our "contents" map and update
   * the state validation object.
   */
  const updateContentAt = (0, _react.useCallback)(function (contentId, content) {
    contents.current[contentId] = content;
    const updatedValidity = {
      [contentId]: content.isValid
    };
    const isValid = updateContentValidity(updatedValidity);
    if (onChange !== undefined) {
      onChange({
        isValid,
        validate,
        getData
      });
    }
  }, [updateContentValidity, onChange, getData, validate]);

  /**
   * When a content unmounts we want to save its current data state so we will be able
   * to provide it as "defaultValue" the next time the component is mounted.
   */
  const saveSnapshotAndRemoveContent = (0, _react.useCallback)(function (contentId) {
    if (contents.current[contentId]) {
      // Merge the data in our stateData
      const updatedData = {
        [contentId]: contents.current[contentId].getData()
      };
      updateContentDataAt(updatedData);

      // Remove the content from our map
      delete contents.current[contentId];
    }
  }, [updateContentDataAt]);
  return {
    getData,
    getSingleContentData,
    validate,
    validation,
    updateContentAt,
    saveSnapshotAndRemoveContent
  };
}