/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.ml;

import java.io.IOException;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Collection;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.component.LifecycleListener;
import org.elasticsearch.core.Strings;
import org.elasticsearch.persistent.PersistentTasksCustomMetadata;
import org.elasticsearch.xpack.core.ml.MlTasks;
import org.elasticsearch.xpack.core.ml.inference.assignment.RoutingInfo;
import org.elasticsearch.xpack.core.ml.inference.assignment.RoutingState;
import org.elasticsearch.xpack.core.ml.inference.assignment.TrainedModelAssignmentMetadata;
import org.elasticsearch.xpack.ml.datafeed.DatafeedRunner;
import org.elasticsearch.xpack.ml.dataframe.DataFrameAnalyticsManager;
import org.elasticsearch.xpack.ml.job.process.autodetect.AutodetectProcessManager;
import org.elasticsearch.xpack.ml.process.MlController;
import org.elasticsearch.xpack.ml.process.MlMemoryTracker;

public class MlLifeCycleService {
    public static final Duration MAX_GRACEFUL_SHUTDOWN_TIME = Duration.of(10L, ChronoUnit.MINUTES);
    private static final Logger logger = LogManager.getLogger(MlLifeCycleService.class);
    private final ClusterService clusterService;
    private final DatafeedRunner datafeedRunner;
    private final MlController mlController;
    private final AutodetectProcessManager autodetectProcessManager;
    private final DataFrameAnalyticsManager analyticsManager;
    private final MlMemoryTracker memoryTracker;
    private final Map<String, Instant> shutdownStartTimes = new ConcurrentHashMap<String, Instant>();

    MlLifeCycleService(ClusterService clusterService, DatafeedRunner datafeedRunner, MlController mlController, AutodetectProcessManager autodetectProcessManager, DataFrameAnalyticsManager analyticsManager, MlMemoryTracker memoryTracker) {
        this.clusterService = Objects.requireNonNull(clusterService);
        this.datafeedRunner = Objects.requireNonNull(datafeedRunner);
        this.mlController = Objects.requireNonNull(mlController);
        this.autodetectProcessManager = Objects.requireNonNull(autodetectProcessManager);
        this.analyticsManager = Objects.requireNonNull(analyticsManager);
        this.memoryTracker = Objects.requireNonNull(memoryTracker);
        clusterService.addLifecycleListener(new LifecycleListener(){

            public void beforeStop() {
                MlLifeCycleService.this.stop();
            }
        });
    }

    public synchronized void stop() {
        try {
            this.analyticsManager.markNodeAsShuttingDown();
            this.datafeedRunner.prepareForImmediateShutdown();
            this.autodetectProcessManager.killAllProcessesOnThisNode();
            this.mlController.stop();
        }
        catch (IOException iOException) {
            // empty catch block
        }
        this.memoryTracker.stop();
    }

    public boolean isNodeSafeToShutdown(String nodeId) {
        return MlLifeCycleService.isNodeSafeToShutdown(nodeId, this.clusterService.state(), this.shutdownStartTimes.get(nodeId), Clock.systemUTC());
    }

    static boolean isNodeSafeToShutdown(String nodeId, ClusterState state, Instant shutdownStartTime, Clock clock) {
        if (shutdownStartTime != null && shutdownStartTime.isBefore(clock.instant().minus(MAX_GRACEFUL_SHUTDOWN_TIME))) {
            return true;
        }
        logger.debug(() -> Strings.format((String)"Checking shutdown safety for node id [%s]", (Object[])new Object[]{nodeId}));
        boolean nodeHasRunningDeployments = MlLifeCycleService.nodeHasRunningDeployments(nodeId, state);
        logger.debug(() -> Strings.format((String)"Node id [%s] has running deployments: %s", (Object[])new Object[]{nodeId, nodeHasRunningDeployments}));
        PersistentTasksCustomMetadata tasks = (PersistentTasksCustomMetadata)state.metadata().getProject().custom("persistent_tasks");
        return MlTasks.nonFailedJobTasksOnNode((PersistentTasksCustomMetadata)tasks, (String)nodeId).isEmpty() && MlTasks.nonFailedSnapshotUpgradeTasksOnNode((PersistentTasksCustomMetadata)tasks, (String)nodeId).isEmpty() && !nodeHasRunningDeployments;
    }

    private static boolean nodeHasRunningDeployments(String nodeId, ClusterState state) {
        TrainedModelAssignmentMetadata metadata = TrainedModelAssignmentMetadata.fromState((ClusterState)state);
        return metadata.allAssignments().values().stream().anyMatch(assignment -> {
            if (assignment.isRoutedToNode(nodeId)) {
                RoutingInfo routingInfo = (RoutingInfo)assignment.getNodeRoutingTable().get(nodeId);
                logger.debug(() -> Strings.format((String)"Assignment deployment id [%s] is routed to shutting down nodeId %s state: %s", (Object[])new Object[]{assignment.getDeploymentId(), nodeId, routingInfo.getState()}));
                return routingInfo.getState().isNoneOf(new RoutingState[]{RoutingState.STOPPED, RoutingState.FAILED});
            }
            return false;
        });
    }

    public void signalGracefulShutdown(Collection<String> shutdownNodeIds) {
        this.signalGracefulShutdown(this.clusterService.state(), shutdownNodeIds, Clock.systemUTC());
    }

    void signalGracefulShutdown(ClusterState state, Collection<String> shutdownNodeIds, Clock clock) {
        String localNodeId = state.nodes().getLocalNodeId();
        this.updateShutdownStartTimes(shutdownNodeIds, localNodeId, clock);
        if (shutdownNodeIds.contains(localNodeId)) {
            this.datafeedRunner.vacateAllDatafeedsOnThisNode("previously assigned node [" + state.nodes().getLocalNode().getName() + "] is shutting down");
            this.autodetectProcessManager.vacateOpenJobsOnThisNode();
        }
    }

    Instant getShutdownStartTime(String nodeId) {
        return this.shutdownStartTimes.get(nodeId);
    }

    private void updateShutdownStartTimes(Collection<String> shutdownNodeIds, String localNodeId, Clock clock) {
        for (String shutdownNodeId : shutdownNodeIds) {
            this.shutdownStartTimes.computeIfAbsent(shutdownNodeId, key -> {
                if (key.equals(localNodeId)) {
                    logger.info("Starting node shutdown sequence for ML");
                }
                return Instant.now(clock);
            });
        }
        this.shutdownStartTimes.keySet().retainAll(shutdownNodeIds);
    }
}

