"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDefendInsightsRoute = void 0;
var _common = require("@kbn/elastic-assistant-common/impl/schemas/common");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _build_response = require("../../lib/build_response");
var _types = require("../../types");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getDefendInsightsRoute = router => {
  router.versioned.get({
    access: 'internal',
    path: _elasticAssistantCommon.DEFEND_INSIGHTS,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution-readWorkflowInsights']
      }
    }
  }).addVersion({
    version: _elasticAssistantCommon.API_VERSIONS.internal.v1,
    validate: {
      request: {
        query: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.DefendInsightsGetRequestQuery)
      },
      response: {
        200: {
          body: {
            custom: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.DefendInsightsGetResponse)
          }
        }
      }
    }
  }, async (context, request, response) => {
    const resp = (0, _build_response.buildResponse)(response);
    const ctx = await context.resolve(['licensing', 'elasticAssistant']);
    const assistantContext = ctx.elasticAssistant;
    const logger = assistantContext.logger;
    try {
      if (!ctx.licensing.license.hasAtLeast('enterprise')) {
        return response.forbidden({
          body: {
            message: 'Your license does not support Automatic Troubleshooting. Please upgrade your license.'
          }
        });
      }
      const dataClient = await assistantContext.getDefendInsightsDataClient();
      const authenticatedUser = await assistantContext.getCurrentUser();
      if (authenticatedUser == null) {
        return resp.error({
          body: `Authenticated user not found`,
          statusCode: 401
        });
      }
      if (!dataClient) {
        return resp.error({
          body: `Defend insights data client not initialized`,
          statusCode: 500
        });
      }
      const defendInsights = await dataClient.findDefendInsightsByParams({
        params: request.query,
        authenticatedUser
      });
      if (defendInsights.length) {
        const agentIds = Array.from(new Set(defendInsights.flatMap(insight => insight.endpointIds)));
        await (0, _helpers.runExternalCallbacks)(_types.CallbackIds.DefendInsightsPostFetch, request, agentIds);
      }
      const updatedDefendInsights = await (0, _helpers.updateDefendInsightsLastViewedAt)({
        dataClient,
        defendInsights,
        authenticatedUser
      });
      return response.ok({
        body: {
          data: updatedDefendInsights
        }
      });
    } catch (err) {
      logger.error(err);
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return resp.error({
        body: {
          success: false,
          error: error.message
        },
        statusCode: error.statusCode
      });
    }
  });
};
exports.getDefendInsightsRoute = getDefendInsightsRoute;