"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getValidateEsql = void 0;
var _fp = require("lodash/fp");
var _securitysolutionUtils = require("@kbn/securitysolution-utils");
var _constants = require("../../../../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getValidateEsql = ({
  logger
}) => {
  return async input => {
    // We want to prevent infinite loops, so we increment the iterations counter for each validation run.
    let error = '';
    try {
      const sanitizedQuery = input.query ? sanitizeQuery(input.query) : '';
      if (!(0, _fp.isEmpty)(sanitizedQuery)) {
        const {
          errors,
          isEsqlQueryAggregating,
          hasMetadataOperator
        } = (0, _securitysolutionUtils.parseEsqlQuery)(sanitizedQuery);
        if (!(0, _fp.isEmpty)(errors)) {
          error = JSON.stringify(errors);
        } else if (!isEsqlQueryAggregating && !hasMetadataOperator) {
          error = `Queries that do't use the STATS...BY function (non-aggregating queries) must include the "metadata _id, _version, _index" operator after the source command. For example: FROM logs* metadata _id, _version, _index.`;
        }
      }
    } catch (err) {
      error = err.message.toString();
      logger.debug(`Error parsing ESQL query: ${error}`);
    }
    return {
      error
    };
  };
};
exports.getValidateEsql = getValidateEsql;
function sanitizeQuery(query) {
  return query.replace(`FROM ${_constants.MISSING_INDEX_PATTERN_PLACEHOLDER}`, 'FROM *') // Replace the index pattern placeholder with a wildcard
  .replaceAll(/\[(macro|lookup):.*?\]/g, '') // Removes any macro or lookup placeholders
  .replaceAll(/\n(\s*?\|\s*?\n)*/g, '\n'); // Removes any empty lines with | (pipe) alone after removing the placeholders
}