"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SiemMigrationsDataMigrationClient = void 0;
var _uuid = require("uuid");
var _siem_migrations_data_base_client = require("./siem_migrations_data_base_client");
var _is_not_found_error = require("../api/util/is_not_found_error");
var _constants = require("./constants");
var _translations = require("../translations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class SiemMigrationsDataMigrationClient extends _siem_migrations_data_base_client.SiemMigrationsDataBaseClient {
  async create(name) {
    const migrationId = (0, _uuid.v4)();
    const index = await this.getIndexName();
    const profileUid = await this.getProfileUid();
    const createdAt = new Date().toISOString();
    await this.esClient.create({
      refresh: 'wait_for',
      id: migrationId,
      index,
      document: {
        name,
        created_by: profileUid,
        created_at: createdAt
      }
    }).catch(error => {
      this.logger.error(`Error creating migration ${migrationId}: ${error}`);
      throw error;
    });
    return migrationId;
  }

  /**
   *
   * Gets the migration document by id or returns undefined if it does not exist.
   *
   * */
  async get(migrationId) {
    const index = await this.getIndexName();
    return this.esClient.get({
      index,
      id: migrationId
    }).then(this.processHit).catch(error => {
      if ((0, _is_not_found_error.isNotFoundError)(error)) {
        return undefined;
      }
      this.logger.error(`Error getting migration ${migrationId}: ${error}`);
      throw error;
    });
  }

  /**
   * Gets all migrations from the index.
   */
  async getAll() {
    const index = await this.getIndexName();
    return this.esClient.search({
      index,
      size: _constants.MAX_ES_SEARCH_SIZE,
      // Adjust size as needed
      query: {
        match_all: {}
      },
      _source: true
    }).then(result => this.processResponseHits(result)).catch(error => {
      this.logger.error(`Error getting all migrations:- ${error}`);
      throw error;
    });
  }

  /**
   *
   * Prepares bulk ES delete operation for a migration document based on its id.
   *
   */
  async prepareDelete(migrationId) {
    const index = await this.getIndexName();
    const migrationDeleteOperation = {
      delete: {
        _index: index,
        _id: migrationId
      }
    };
    return [migrationDeleteOperation];
  }

  /**
   * Saves a migration as started, updating the last execution parameters with the current timestamp.
   */
  async saveAsStarted({
    id,
    connectorId
  }) {
    await this.updateLastExecution(id, {
      started_at: new Date().toISOString(),
      connector_id: connectorId,
      is_stopped: false,
      error: null,
      finished_at: null
    });
  }

  /**
   * Saves a migration as ended, updating the last execution parameters with the current timestamp.
   */
  async saveAsFinished({
    id,
    error
  }) {
    var _currentMigrationDoc$, _currentMigrationDoc$2, _currentMigrationDoc$3;
    const currentMigrationDoc = await this.get(id);
    if (!currentMigrationDoc) {
      throw new Error((0, _translations.MIGRATION_ID_NOT_FOUND)(id));
    }
    const finishedAt = new Date();
    let latestExecutionTimeMs = 0;

    // ideally started_at should always be defined here, but if it is not, we do not want to throw an error
    if ((_currentMigrationDoc$ = currentMigrationDoc.last_execution) !== null && _currentMigrationDoc$ !== void 0 && _currentMigrationDoc$.started_at) {
      latestExecutionTimeMs = finishedAt.getTime() - new Date(currentMigrationDoc.last_execution.started_at).getTime();
    }
    const currentExecutionTime = (_currentMigrationDoc$2 = currentMigrationDoc === null || currentMigrationDoc === void 0 ? void 0 : (_currentMigrationDoc$3 = currentMigrationDoc.last_execution) === null || _currentMigrationDoc$3 === void 0 ? void 0 : _currentMigrationDoc$3.total_execution_time_ms) !== null && _currentMigrationDoc$2 !== void 0 ? _currentMigrationDoc$2 : 0;
    await this.updateLastExecution(id, {
      finished_at: finishedAt.toISOString(),
      total_execution_time_ms: currentExecutionTime + latestExecutionTimeMs,
      error
    });
  }

  /**
   * Saves a migration as failed, updating the last execution parameters with the provided error message.
   */
  async saveAsFailed({
    id,
    error
  }) {
    await this.saveAsFinished({
      id,
      error
    });
  }

  /**
   * Sets `is_stopped` flag for migration document.
   * It does not update `finished_at` timestamp, `saveAsFinished` or `saveAsFailed` should be called separately.
   */
  async setIsStopped({
    id
  }) {
    await this.updateLastExecution(id, {
      is_stopped: true
    });
  }

  /**
   * Updates the last execution parameters for a migration document.
   */
  async updateLastExecution(id, lastExecutionParams) {
    const index = await this.getIndexName();
    const doc = {
      last_execution: lastExecutionParams
    };
    await this.esClient.update({
      index,
      id,
      refresh: 'wait_for',
      doc,
      retry_on_conflict: 1
    }).catch(error => {
      this.logger.error(`Error updating last execution for migration ${id}: ${error}`);
      throw error;
    });
  }

  /**
   * Updates the migration document with the provided values.
   */
  async update(id, doc) {
    const index = await this.getIndexName();
    await this.esClient.update({
      index,
      id,
      doc,
      refresh: 'wait_for',
      retry_on_conflict: 1
    }).catch(error => {
      this.logger.error(`Error updating migration: ${error}`);
      throw error;
    });
  }
}
exports.SiemMigrationsDataMigrationClient = SiemMigrationsDataMigrationClient;