"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SiemMigrationsDataLookupsClient = void 0;
var _jsSha = require("js-sha256");
var _indexAdapter = require("@kbn/index-adapter");
var _constants = require("../../../../../common/siem_migrations/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class SiemMigrationsDataLookupsClient {
  constructor(currentUser, esScopedClient, logger, spaceId) {
    this.currentUser = currentUser;
    this.esScopedClient = esScopedClient;
    this.logger = logger;
    this.spaceId = spaceId;
  }
  async create(lookupName, data) {
    const indexName = `${_constants.LOOKUPS_INDEX_PREFIX}${this.spaceId}_${lookupName}`;
    try {
      await this.executeEs(() => this.esScopedClient.asCurrentUser.indices.create({
        index: indexName,
        settings: {
          index: {
            mode: 'lookup'
          }
        },
        mappings: {
          dynamic: 'runtime'
        }
      }));
    } catch (error) {
      var _error$meta, _error$meta$body, _error$meta$body$erro;
      if ((error === null || error === void 0 ? void 0 : (_error$meta = error.meta) === null || _error$meta === void 0 ? void 0 : (_error$meta$body = _error$meta.body) === null || _error$meta$body === void 0 ? void 0 : (_error$meta$body$erro = _error$meta$body.error) === null || _error$meta$body$erro === void 0 ? void 0 : _error$meta$body$erro.type) !== 'resource_already_exists_exception') {
        this.logger.error(`Error creating lookup index ${indexName} - ${error.message}`);
        throw error;
      }
    }
    if (data.length > 0) {
      await this.indexData(indexName, data);
    }
    return indexName;
  }
  async indexData(indexName, data) {
    const body = data.flatMap(doc => [{
      create: {
        _index: indexName,
        _id: this.generateDocumentHash(doc)
      }
    }, doc]);
    try {
      await this.executeEs(() => this.esScopedClient.asCurrentUser.bulk({
        index: indexName,
        body
      }));
    } catch (error) {
      if ((error === null || error === void 0 ? void 0 : error.statusCode) !== 404) {
        this.logger.error(`Error indexing data for lookup index ${indexName} - ${error.message}`);
        throw error;
      }
    }
  }
  async executeEs(fn) {
    return (0, _indexAdapter.retryTransientEsErrors)(fn, {
      logger: this.logger
    });
  }
  generateDocumentHash(document) {
    return _jsSha.sha256.create().update(JSON.stringify(document)).hex(); // document need to be created in a deterministic way
  }
}
exports.SiemMigrationsDataLookupsClient = SiemMigrationsDataLookupsClient;