"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.startPrivilegeMonitoringTask = exports.scheduleNow = exports.removePrivilegeMonitoringTask = exports.registerPrivilegeMonitoringTask = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/core/server");
var _moment = _interopRequireDefault(require("moment"));
var _constants = require("../constants");
var _state = require("./state");
var _api_key = require("../auth/api_key");
var _data_client = require("../engine/data_client");
var _data_sources_service = require("../data_sources/data_sources_service");
var _helpers = require("../../risk_score/tasks/helpers");
var _saved_object = require("../auth/saved_object");
var _saved_objects = require("../saved_objects");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class EngineAlreadyRunningError extends Error {
  constructor() {
    super('The monitoring engine is already running');
    (0, _defineProperty2.default)(this, "statusCode", void 0);
    this.statusCode = 409;
  }
}
const getTaskName = () => _constants.TYPE;
const getTaskId = namespace => `${_constants.TYPE}:${namespace}:${_constants.VERSION}`;
const registerPrivilegeMonitoringTask = ({
  getStartServices,
  logger,
  telemetry,
  taskManager,
  kibanaVersion,
  experimentalFeatures,
  config
}) => {
  if (!taskManager) {
    logger.info('[Privilege Monitoring]  Task Manager is unavailable; skipping privilege monitoring task registration.');
    return;
  }
  const getPrivilegedUserMonitoringDataClient = async namespace => {
    const [core, {
      taskManager: taskManagerStart,
      security,
      encryptedSavedObjects
    }] = await getStartServices();
    const apiKeyManager = (0, _api_key.getApiKeyManager)({
      core,
      logger,
      security,
      encryptedSavedObjects,
      namespace
    });
    const client = await apiKeyManager.getClient();
    if (!client) {
      logger.error('[Privilege Monitoring] Unable to create Elasticsearch client from API key.');
      return undefined;
    }
    return new _data_client.PrivilegeMonitoringDataClient({
      logger,
      clusterClient: client.clusterClient,
      namespace,
      taskManager: taskManagerStart,
      savedObjects: core.savedObjects,
      auditLogger: core.security.audit.withoutRequest,
      experimentalFeatures,
      kibanaVersion,
      telemetry,
      apiKeyManager
    });
  };
  taskManager.registerTaskDefinitions({
    [getTaskName()]: {
      title: 'Entity Analytics Privilege Monitoring',
      timeout: _constants.TIMEOUT,
      stateSchemaByVersion: _state.stateSchemaByVersion,
      createTaskRunner: createPrivilegeMonitoringTaskRunnerFactory({
        logger,
        telemetry,
        experimentalFeatures,
        getStartServices,
        getPrivilegedUserMonitoringDataClient,
        config
      })
    }
  });
};
exports.registerPrivilegeMonitoringTask = registerPrivilegeMonitoringTask;
const createPrivilegeMonitoringTaskRunnerFactory = deps => ({
  taskInstance
}) => {
  let cancelled = false;
  const isCancelled = () => cancelled;
  return {
    run: async () => {
      const [core] = await deps.getStartServices();
      const config = deps.config;
      return runPrivilegeMonitoringTask({
        isCancelled,
        logger: deps.logger,
        telemetry: deps.telemetry,
        taskInstance,
        experimentalFeatures: deps.experimentalFeatures,
        core,
        config,
        getPrivilegedUserMonitoringDataClient: deps.getPrivilegedUserMonitoringDataClient
      });
    },
    cancel: async () => {
      cancelled = true;
    }
  };
};
const startPrivilegeMonitoringTask = async ({
  logger,
  namespace,
  taskManager
}) => {
  const taskId = getTaskId(namespace);
  try {
    await taskManager.ensureScheduled({
      id: taskId,
      taskType: getTaskName(),
      scope: _constants.SCOPE,
      schedule: {
        interval: _constants.INTERVAL
      },
      state: {
        ..._state.defaultState,
        namespace
      },
      params: {
        version: _constants.VERSION
      }
    });
    logger.info(`Scheduling privilege monitoring task with id ${taskId}`);
  } catch (e) {
    logger.warn(`[Privilege Monitoring]  [task ${taskId}]: error scheduling task, received ${e.message}`);
    throw e;
  }
};
exports.startPrivilegeMonitoringTask = startPrivilegeMonitoringTask;
const runPrivilegeMonitoringTask = async ({
  isCancelled,
  logger,
  taskInstance,
  getPrivilegedUserMonitoringDataClient,
  core,
  config
}) => {
  const state = taskInstance.state;
  const taskStartTime = (0, _moment.default)().utc().toISOString();
  const updatedState = {
    lastExecutionTimestamp: taskStartTime,
    namespace: state.namespace,
    runs: state.runs + 1
  };
  if (isCancelled()) {
    logger.info('[Privilege Monitoring] Task was cancelled.');
    return {
      state: updatedState
    };
  }
  try {
    logger.info('[Privilege Monitoring] Running privilege monitoring task');
    const dataClient = await getPrivilegedUserMonitoringDataClient(state.namespace);
    if (!dataClient) {
      logger.error('[Privilege Monitoring] error creating data client.');
      throw Error('No data client was found');
    }
    const maxUsersAllowed = config.entityAnalytics.monitoring.privileges.users.maxPrivilegedUsersAllowed;
    const request = (0, _helpers.buildFakeScopedRequest)({
      namespace: state.namespace,
      coreStart: core
    });
    const soClient = core.savedObjects.getScopedClient(request, {
      includedHiddenTypes: [_saved_object.PrivilegeMonitoringApiKeyType.name, _saved_objects.monitoringEntitySourceType.name],
      excludedExtensions: [_server.SECURITY_EXTENSION_ID]
    });
    const dataSourcesService = (0, _data_sources_service.createDataSourcesService)(dataClient, soClient, maxUsersAllowed);
    await dataSourcesService.syncAllSources();
  } catch (e) {
    logger.error(`[Privilege Monitoring] Error running privilege monitoring task: ${e.message}`);
  }
  logger.info('[Privilege Monitoring] Finished running privilege monitoring task');
  return {
    state: updatedState
  };
};
const removePrivilegeMonitoringTask = async ({
  taskManager,
  namespace,
  logger
}) => {
  const taskId = getTaskId(namespace);
  try {
    await taskManager.removeIfExists(taskId);
    logger.info(`Removed privilege monitoring task with id ${taskId}`);
  } catch (e) {
    logger.warn(`[Privilege Monitoring][task ${taskId}]: error removing task, received ${e.message}`);
    throw e;
  }
};
exports.removePrivilegeMonitoringTask = removePrivilegeMonitoringTask;
const scheduleNow = async ({
  logger,
  namespace,
  taskManager
}) => {
  const taskId = getTaskId(namespace);
  logger.info(`[Privilege Monitoring][task ${taskId}]: Attempting to schedule task to run now`);
  try {
    return taskManager.runSoon(taskId);
  } catch (e) {
    logger.warn(`[Privilege Monitoring][task ${taskId}]: error scheduling task now, received '${e.message}'`);
    if (e.message.contains('as it is currently running')) {
      throw new EngineAlreadyRunningError();
    }
    throw e;
  }
};
exports.scheduleNow = scheduleNow;