"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PrivilegeMonitoringEngineDescriptorClient = exports.MAX_PER_PAGE = void 0;
var _privilege_monitoring_type = require("./privilege_monitoring_type");
var _constants = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_PER_PAGE = exports.MAX_PER_PAGE = 10_000;
class PrivilegeMonitoringEngineDescriptorClient {
  constructor(deps) {
    this.deps = deps;
  }
  getSavedObjectId() {
    return `privilege-monitoring-${this.deps.namespace}`;
  }
  async init() {
    const engineDescriptor = await this.find();
    if (engineDescriptor.total === 1) {
      return this.updateExistingDescriptor(engineDescriptor);
    }
    const {
      attributes
    } = await this.deps.soClient.create(_privilege_monitoring_type.privilegeMonitoringTypeName, {
      status: _constants.PRIVILEGE_MONITORING_ENGINE_STATUS.STARTED
    }, {
      id: this.getSavedObjectId(),
      refresh: 'wait_for'
    });
    return attributes;
  }
  async updateExistingDescriptor(engineDescriptor) {
    const old = engineDescriptor.saved_objects[0].attributes;
    const update = {
      ...old,
      error: undefined,
      status: _constants.PRIVILEGE_MONITORING_ENGINE_STATUS.STARTED,
      apiKey: ''
    };
    await this.deps.soClient.update(_privilege_monitoring_type.privilegeMonitoringTypeName, this.getSavedObjectId(), update, {
      refresh: 'wait_for'
    });
    return update;
  }
  async update(engine) {
    const id = this.getSavedObjectId();
    const {
      attributes
    } = await this.deps.soClient.update(_privilege_monitoring_type.privilegeMonitoringTypeName, id, engine, {
      refresh: 'wait_for'
    });
    return attributes;
  }
  async updateStatus(status) {
    return this.update({
      status
    });
  }
  async find() {
    return this.deps.soClient.find({
      type: _privilege_monitoring_type.privilegeMonitoringTypeName,
      namespaces: [this.deps.namespace],
      perPage: MAX_PER_PAGE
    });
  }
  async get() {
    const so = await this.maybeGet();
    if (!so) {
      return {
        status: _constants.PRIVILEGE_MONITORING_ENGINE_STATUS.NOT_INSTALLED,
        error: undefined
      };
    }
    return so.attributes;
  }
  async maybeGet() {
    try {
      const result = await this.deps.soClient.get(_privilege_monitoring_type.privilegeMonitoringTypeName, this.getSavedObjectId());
      return result;
    } catch (e) {
      if (e.output && e.output.statusCode === 404) {
        return undefined;
      }
      throw e;
    }
  }
  async getStatus() {
    const engineDescriptor = await this.get();
    return {
      status: engineDescriptor.status,
      error: engineDescriptor.error
    };
  }
  async delete() {
    const id = this.getSavedObjectId();
    return this.deps.soClient.delete(_privilege_monitoring_type.privilegeMonitoringTypeName, id, {
      refresh: 'wait_for'
    });
  }
}
exports.PrivilegeMonitoringEngineDescriptorClient = PrivilegeMonitoringEngineDescriptorClient;