"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.patchRuleRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _zodHelpers = require("@kbn/zod-helpers");
var _rule_management = require("../../../../../../../common/api/detection_engine/rule_management");
var _constants = require("../../../../../../../common/constants");
var _utils = require("../../../../routes/utils");
var _read_rules = require("../../../logic/detection_rules_client/read_rules");
var _check_for_default_rule_exception_list = require("../../../logic/exceptions/check_for_default_rule_exception_list");
var _validate_rule_default_exception_list = require("../../../logic/exceptions/validate_rule_default_exception_list");
var _utils2 = require("../../../utils/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const patchRuleRoute = router => {
  router.versioned.patch({
    access: 'public',
    path: _constants.DETECTION_ENGINE_RULES_URL,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: {
        // Use non-exact validation because everything is optional in patch - since everything is optional,
        // io-ts can't find the right schema from the type specific union and the exact check breaks.
        // We do type specific validation after fetching the existing rule so we know the rule type.
        body: (0, _zodHelpers.buildRouteValidationWithZod)(_rule_management.PatchRuleRequestBody)
      }
    }
  }, async (context, request, response) => {
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    const validationErrors = (0, _rule_management.validatePatchRuleRequestBody)(request.body);
    if (validationErrors.length) {
      return siemResponse.error({
        statusCode: 400,
        body: validationErrors
      });
    }
    try {
      const params = request.body;
      const rulesClient = await (await context.alerting).getRulesClient();
      const detectionRulesClient = (await context.securitySolution).getDetectionRulesClient();
      const existingRule = await (0, _read_rules.readRules)({
        rulesClient,
        ruleId: params.rule_id,
        id: params.id
      });
      if (!existingRule) {
        const error = (0, _utils2.getIdError)({
          id: params.id,
          ruleId: params.rule_id
        });
        return siemResponse.error({
          body: error.message,
          statusCode: error.statusCode
        });
      }
      (0, _check_for_default_rule_exception_list.checkDefaultRuleExceptionListReferences)({
        exceptionLists: params.exceptions_list
      });
      await (0, _validate_rule_default_exception_list.validateRuleDefaultExceptionList)({
        exceptionsList: params.exceptions_list,
        rulesClient,
        ruleRuleId: params.rule_id,
        ruleId: params.id
      });
      const patchedRule = await detectionRulesClient.patchRule({
        rulePatch: params
      });
      return response.ok({
        body: patchedRule
      });
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return siemResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.patchRuleRoute = patchRuleRoute;