"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createPolicyDataStreamsIfNeeded = void 0;
var _pMap = _interopRequireDefault(require("p-map"));
var _index_name_utilities = require("../../../common/endpoint/utils/index_name_utilities");
var _utils = require("../../endpoint/utils");
var _simple_mem_cache = require("../../endpoint/lib/simple_mem_cache");
var _constants = require("../../../common/endpoint/constants");
var _stringify = require("../../endpoint/utils/stringify");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const cache = new _simple_mem_cache.SimpleMemCache({
  // Cache of created Datastreams last for 12h, at which point it is checked again.
  // This is just a safeguard case (for whatever reason) the index is deleted
  // 1.8e+7 ===  hours
  ttl: 1.8e7
});
/**
 * Ensures that the DOT index Datastreams necessary to support Elastic Defend are crated (prior to
 * endpoint writing data to them)
 */
const createPolicyDataStreamsIfNeeded = async ({
  endpointServices,
  endpointPolicyIds
}) => {
  const logger = endpointServices.createLogger('endpointPolicyDatastreamCreator');
  const esClient = endpointServices.getInternalEsClient();
  logger.debug(() => `Checking if datastreams need to be created for Endpoint integration policy [${endpointPolicyIds.join(', ')}]`);
  const fleetServices = endpointServices.getInternalFleetServices(undefined, true);
  const policyNamespaces = await fleetServices.getPolicyNamespace({
    integrationPolicies: endpointPolicyIds
  });
  const indexesCreated = [];
  const createErrors = [];
  const indicesToCreate = Array.from(Object.values(policyNamespaces.integrationPolicy).reduce((acc, namespaceList) => {
    for (const namespace of namespaceList) {
      acc.add((0, _index_name_utilities.buildIndexNameWithNamespace)(_constants.DEFAULT_DIAGNOSTIC_INDEX_PATTERN, namespace));
      acc.add((0, _index_name_utilities.buildIndexNameWithNamespace)(_constants.ENDPOINT_ACTION_RESPONSES_DS, namespace));
      if (endpointServices.isServerless()) {
        acc.add((0, _index_name_utilities.buildIndexNameWithNamespace)(_constants.ENDPOINT_HEARTBEAT_INDEX_PATTERN, namespace));
      }
    }
    return acc;
  }, new Set()));
  const processesDatastreamIndex = async datastreamIndexName => {
    if (cache.get(datastreamIndexName)) {
      return;
    }
    const doesDataStreamAlreadyExist = await esClient.indices.exists({
      index: datastreamIndexName
    }).catch(_utils.catchAndWrapError);
    if (doesDataStreamAlreadyExist) {
      cache.set(datastreamIndexName, true);
      return;
    }
    await esClient.indices.createDataStream({
      name: datastreamIndexName
    }).then(() => {
      indexesCreated.push(datastreamIndexName);
      cache.set(datastreamIndexName, true);
    }).catch(err => {
      var _err$body, _err$body$error, _err$body$error2, _err$body2;
      // It's possible that between the `.exists()` check and this `.createDataStream()` that
      // the index could have been created. If that's the case, then just ignore the error.
      if (((_err$body = err.body) === null || _err$body === void 0 ? void 0 : (_err$body$error = _err$body.error) === null || _err$body$error === void 0 ? void 0 : _err$body$error.type) === 'resource_already_exists_exception') {
        cache.set(datastreamIndexName, true);
        return;
      }
      createErrors.push(`Attempt to create datastream [${datastreamIndexName}] failed:\n${(0, _stringify.stringify)((_err$body$error2 = (_err$body2 = err.body) === null || _err$body2 === void 0 ? void 0 : _err$body2.error) !== null && _err$body$error2 !== void 0 ? _err$body$error2 : err)}`);
    });
  };
  logger.debug(() => `Checking if the following datastream(s) need to be created:\n    ${indicesToCreate.join('\n    ')}`);
  await (0, _pMap.default)(indicesToCreate, processesDatastreamIndex, {
    concurrency: 10
  });
  if (indexesCreated.length > 0) {
    logger.info(`Datastream(s) created in support of Elastic Defend policy [${endpointPolicyIds.join(', ')}]:\n    ${indexesCreated.join('\n    ')}`);
  } else if (createErrors.length === 0) {
    logger.debug(() => `Nothing to do. Datastreams already exist`);
  }
  if (createErrors.length > 0) {
    logger.error(`${createErrors.length} errors encountered:\n${createErrors.join('\n--------\n')}`);
  }
};
exports.createPolicyDataStreamsIfNeeded = createPolicyDataStreamsIfNeeded;
createPolicyDataStreamsIfNeeded.cache = cache;