"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.upsertMigrationResources = exports.updateMigrationRules = exports.updateMigration = exports.updateIndexPattern = exports.stopRuleMigration = exports.startRuleMigration = exports.installMigrationRules = exports.getRuleMigrationsStatsAll = exports.getRuleMigrationsPrebuiltRules = exports.getRuleMigrationTranslationStats = exports.getRuleMigrationStats = exports.getRuleMigrationMissingPrivileges = exports.getMissingResources = exports.getMigrationRules = exports.getIntegrationsStats = exports.getIntegrations = exports.deleteMigration = exports.createRuleMigration = exports.addRulesToMigration = void 0;
var _shared = require("@kbn/openapi-common/shared");
var _kibana = require("../../../common/lib/kibana");
var _constants = require("../../../../common/siem_migrations/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/** Retrieves the stats for the specific migration. */
const getRuleMigrationStats = async ({
  migrationId,
  signal
}) => {
  // Typed with `RuleMigrationStats` instead of `GetRuleMigrationStatsResponse` to use native enums instead of the zod enum
  return _kibana.KibanaServices.get().http.get((0, _shared.replaceParams)(_constants.SIEM_RULE_MIGRATION_STATS_PATH, {
    migration_id: migrationId
  }), {
    version: '1',
    signal
  });
};

/** Retrieves the stats for all the existing migrations, aggregated by `migration_id`. */
exports.getRuleMigrationStats = getRuleMigrationStats;
const getRuleMigrationsStatsAll = async ({
  signal
} = {}) => {
  // Typed with `RuleMigrationStats` instead of `GetAllStatsRuleMigrationResponse` to use native enums instead of the zod enum
  return _kibana.KibanaServices.get().http.get(_constants.SIEM_RULE_MIGRATIONS_ALL_STATS_PATH, {
    version: '1',
    signal
  });
};
exports.getRuleMigrationsStatsAll = getRuleMigrationsStatsAll;
/** Starts a new migration with the provided rules. */
const createRuleMigration = async ({
  signal,
  name
}) => {
  return _kibana.KibanaServices.get().http.put(_constants.SIEM_RULE_MIGRATIONS_PATH, {
    version: '1',
    signal,
    body: JSON.stringify({
      name
    })
  });
};
exports.createRuleMigration = createRuleMigration;
/** Adds provided rules to an existing migration */
const addRulesToMigration = async ({
  migrationId,
  body,
  signal
}) => {
  return _kibana.KibanaServices.get().http.post((0, _shared.replaceParams)(_constants.SIEM_RULE_MIGRATION_RULES_PATH, {
    migration_id: migrationId
  }), {
    body: JSON.stringify(body),
    version: '1',
    signal
  });
};
exports.addRulesToMigration = addRulesToMigration;
/** Retrieves all missing resources of a specific migration. */
const getMissingResources = async ({
  migrationId,
  signal
}) => {
  return _kibana.KibanaServices.get().http.get((0, _shared.replaceParams)(_constants.SIEM_RULE_MIGRATION_RESOURCES_MISSING_PATH, {
    migration_id: migrationId
  }), {
    version: '1',
    signal
  });
};
exports.getMissingResources = getMissingResources;
/** Updates or creates resources for a specific migration. */
const upsertMigrationResources = async ({
  migrationId,
  body,
  signal
}) => {
  return _kibana.KibanaServices.get().http.post((0, _shared.replaceParams)(_constants.SIEM_RULE_MIGRATION_RESOURCES_PATH, {
    migration_id: migrationId
  }), {
    body: JSON.stringify(body),
    version: '1',
    signal
  });
};
exports.upsertMigrationResources = upsertMigrationResources;
/** Starts a new migration with the provided rules. */
const startRuleMigration = async ({
  migrationId,
  retry,
  langSmithOptions,
  signal,
  settings: {
    connectorId,
    skipPrebuiltRulesMatching = false
  }
}) => {
  const body = {
    settings: {
      connector_id: connectorId,
      skip_prebuilt_rules_matching: skipPrebuiltRulesMatching
    },
    retry,
    langsmith_options: langSmithOptions
  };
  return _kibana.KibanaServices.get().http.post((0, _shared.replaceParams)(_constants.SIEM_RULE_MIGRATION_START_PATH, {
    migration_id: migrationId
  }), {
    body: JSON.stringify(body),
    version: '1',
    signal
  });
};
exports.startRuleMigration = startRuleMigration;
/** Stops a new migration with the provided rules. */
const stopRuleMigration = async ({
  migrationId,
  signal
}) => {
  return _kibana.KibanaServices.get().http.post((0, _shared.replaceParams)(_constants.SIEM_RULE_MIGRATION_STOP_PATH, {
    migration_id: migrationId
  }), {
    version: '1',
    signal
  });
};
exports.stopRuleMigration = stopRuleMigration;
/** Retrieves all the migration rule documents of a specific migration. */
const getMigrationRules = async ({
  migrationId,
  page,
  perPage,
  sortField,
  sortDirection,
  filters,
  signal
}) => {
  return _kibana.KibanaServices.get().http.get((0, _shared.replaceParams)(_constants.SIEM_RULE_MIGRATION_RULES_PATH, {
    migration_id: migrationId
  }), {
    version: '1',
    query: {
      page,
      per_page: perPage,
      sort_field: sortField,
      sort_direction: sortDirection,
      search_term: filters === null || filters === void 0 ? void 0 : filters.searchTerm,
      ids: filters === null || filters === void 0 ? void 0 : filters.ids,
      is_prebuilt: filters === null || filters === void 0 ? void 0 : filters.prebuilt,
      is_installed: filters === null || filters === void 0 ? void 0 : filters.installed,
      is_fully_translated: filters === null || filters === void 0 ? void 0 : filters.fullyTranslated,
      is_partially_translated: filters === null || filters === void 0 ? void 0 : filters.partiallyTranslated,
      is_untranslatable: filters === null || filters === void 0 ? void 0 : filters.untranslatable,
      is_failed: filters === null || filters === void 0 ? void 0 : filters.failed,
      is_missing_index: filters === null || filters === void 0 ? void 0 : filters.missingIndex
    },
    signal
  });
};
exports.getMigrationRules = getMigrationRules;
/** Retrieves all the migration rule documents of a specific migration. */
const getRuleMigrationMissingPrivileges = async ({
  signal
}) => {
  return _kibana.KibanaServices.get().http.get(_constants.SIEM_RULE_MIGRATION_MISSING_PRIVILEGES_PATH, {
    version: '1',
    signal
  });
};
exports.getRuleMigrationMissingPrivileges = getRuleMigrationMissingPrivileges;
/**
 * Retrieves the translation stats for the migration.
 */
const getRuleMigrationTranslationStats = async ({
  migrationId,
  signal
}) => {
  return _kibana.KibanaServices.get().http.get((0, _shared.replaceParams)(_constants.SIEM_RULE_MIGRATION_TRANSLATION_STATS_PATH, {
    migration_id: migrationId
  }), {
    version: '1',
    signal
  });
};
exports.getRuleMigrationTranslationStats = getRuleMigrationTranslationStats;
/** Installs the provided rule ids for a specific migration. */
const installMigrationRules = async ({
  migrationId,
  ids,
  enabled,
  signal
}) => {
  return _kibana.KibanaServices.get().http.post((0, _shared.replaceParams)(_constants.SIEM_RULE_MIGRATION_INSTALL_PATH, {
    migration_id: migrationId
  }), {
    version: '1',
    body: JSON.stringify({
      ids,
      enabled
    }),
    signal
  });
};
exports.installMigrationRules = installMigrationRules;
/** Retrieves all prebuilt rules matched within a specific migration. */
const getRuleMigrationsPrebuiltRules = async ({
  migrationId,
  signal
}) => {
  return _kibana.KibanaServices.get().http.get((0, _shared.replaceParams)(_constants.SIEM_RULE_MIGRATIONS_PREBUILT_RULES_PATH, {
    migration_id: migrationId
  }), {
    version: '1',
    signal
  });
};
exports.getRuleMigrationsPrebuiltRules = getRuleMigrationsPrebuiltRules;
/** Retrieves existing integrations. */
const getIntegrations = async ({
  signal
} = {}) => {
  return _kibana.KibanaServices.get().http.get(_constants.SIEM_RULE_MIGRATIONS_INTEGRATIONS_PATH, {
    version: '1',
    signal
  });
};
exports.getIntegrations = getIntegrations;
/** Retrieves existing integrations. */
const getIntegrationsStats = async ({
  signal
} = {}) => {
  return _kibana.KibanaServices.get().http.get(_constants.SIEM_RULE_MIGRATIONS_INTEGRATIONS_STATS_PATH, {
    version: '1',
    signal
  });
};
exports.getIntegrationsStats = getIntegrationsStats;
/** Updates provided migration rules. */
const updateMigrationRules = async ({
  migrationId,
  rulesToUpdate,
  signal
}) => {
  return _kibana.KibanaServices.get().http.patch((0, _shared.replaceParams)(_constants.SIEM_RULE_MIGRATION_RULES_PATH, {
    migration_id: migrationId
  }), {
    version: '1',
    body: JSON.stringify(rulesToUpdate),
    signal
  });
};
exports.updateMigrationRules = updateMigrationRules;
const updateMigration = async ({
  migrationId,
  signal,
  body
}) => {
  return _kibana.KibanaServices.get().http.patch((0, _shared.replaceParams)(_constants.SIEM_RULE_MIGRATION_PATH, {
    migration_id: migrationId
  }), {
    version: '1',
    body: JSON.stringify(body),
    signal
  });
};
exports.updateMigration = updateMigration;
const deleteMigration = async ({
  migrationId,
  signal
}) => {
  return _kibana.KibanaServices.get().http.delete((0, _shared.replaceParams)(_constants.SIEM_RULE_MIGRATION_PATH, {
    migration_id: migrationId
  }), {
    version: '1',
    signal
  });
};
exports.deleteMigration = deleteMigration;
const updateIndexPattern = async ({
  migrationId,
  indexPattern,
  ids,
  signal
}) => {
  const payload = ids !== null && ids !== void 0 && ids.length ? {
    index_pattern: indexPattern,
    ids
  } : {
    index_pattern: indexPattern
  };
  return _kibana.KibanaServices.get().http.post((0, _shared.replaceParams)(_constants.SIEM_RULE_MIGRATION_UPDATE_INDEX_PATTERN_PATH, {
    migration_id: migrationId
  }), {
    version: '1',
    body: JSON.stringify(payload),
    signal
  });
};
exports.updateIndexPattern = updateIndexPattern;