"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useUserLimitStatus = exports.usePrivilegedMonitoringHealth = exports.usePrivilegedMonitoringEngineStatus = exports.computeUserStats = void 0;
var _reactQuery = require("@tanstack/react-query");
var _api = require("../api/api");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Helper function to compute user statistics from health data
 */
const computeUserStats = data => {
  if (!(data !== null && data !== void 0 && data.users)) {
    return null;
  }
  const {
    current_count: currentCount,
    max_allowed: maxAllowed
  } = data.users;
  return {
    currentCount,
    maxAllowed,
    isLimitExceeded: currentCount > maxAllowed,
    isNearLimit: currentCount >= maxAllowed * 0.9,
    // 90% threshold
    remainingSlots: Math.max(0, maxAllowed - currentCount),
    usagePercentage: Math.round(currentCount / maxAllowed * 100)
  };
};
exports.computeUserStats = computeUserStats;
const usePrivilegedMonitoringHealth = options => {
  const {
    fetchPrivilegeMonitoringEngineStatus
  } = (0, _api.useEntityAnalyticsRoutes)();
  const {
    data,
    isLoading,
    isError,
    error
  } = (0, _reactQuery.useQuery)({
    queryKey: ['GET', 'PRIVILEGED_MONITORING_HEALTH'],
    queryFn: fetchPrivilegeMonitoringEngineStatus,
    retry: 0,
    enabled: options === null || options === void 0 ? void 0 : options.enabled,
    refetchInterval: 30000 // Refresh every 30 seconds to keep user count current
  });
  const userStats = computeUserStats(data);
  return {
    // Raw API data
    healthData: data,
    isLoading,
    isError,
    error,
    // Computed user stats
    userStats
  };
};
exports.usePrivilegedMonitoringHealth = usePrivilegedMonitoringHealth;
const usePrivilegedMonitoringEngineStatus = () => {
  const {
    healthData,
    isLoading,
    isError,
    error
  } = usePrivilegedMonitoringHealth();
  return {
    data: healthData,
    isLoading,
    isError,
    error
  };
};

// Hook specifically for user limit functionality
exports.usePrivilegedMonitoringEngineStatus = usePrivilegedMonitoringEngineStatus;
const useUserLimitStatus = options => {
  const {
    data,
    isLoading,
    isError,
    error
  } = (0, _reactQuery.useQuery)({
    queryKey: ['GET', 'PRIVILEGED_MONITORING_HEALTH'],
    queryFn: (0, _api.useEntityAnalyticsRoutes)().fetchPrivilegeMonitoringEngineStatus,
    retry: 0,
    enabled: options === null || options === void 0 ? void 0 : options.enabled,
    refetchInterval: 30000
  });
  const userStats = computeUserStats(data);
  return {
    userStats,
    isLoading,
    isError,
    error
  };
};
exports.useUserLimitStatus = useUserLimitStatus;