"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerIndicesRoutes = registerIndicesRoutes;
var _configSchema = require("@kbn/config-schema");
var _i18n = require("@kbn/i18n");
var _types = require("@kbn/search-index-documents/types");
var _lib = require("@kbn/search-index-documents/lib");
var _routes = require("../../common/routes");
var _indices = require("../lib/indices");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerIndicesRoutes(router, logger) {
  router.post({
    path: _routes.POST_CREATE_INDEX_ROUTE,
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {
      body: _configSchema.schema.object({
        indexName: _configSchema.schema.string()
      })
    },
    options: {
      access: 'internal'
    }
  }, async (context, request, response) => {
    const core = await context.core;
    const client = core.elasticsearch.client.asCurrentUser;
    const data = request.body;
    try {
      const body = await (0, _indices.createIndex)(client, logger, data);
      return response.ok({
        body,
        headers: {
          'content-type': 'application/json'
        }
      });
    } catch (e) {
      var _e$meta, _e$meta$body, _e$meta$body$error, _e$meta2, _e$meta3;
      switch (e === null || e === void 0 ? void 0 : (_e$meta = e.meta) === null || _e$meta === void 0 ? void 0 : (_e$meta$body = _e$meta.body) === null || _e$meta$body === void 0 ? void 0 : (_e$meta$body$error = _e$meta$body.error) === null || _e$meta$body$error === void 0 ? void 0 : _e$meta$body$error.type) {
        case 'resource_already_exists_exception':
          return response.conflict({
            body: {
              message: e.message
            }
          });
      }
      return response.customError({
        statusCode: e !== null && e !== void 0 && e.meta && (_e$meta2 = e.meta) !== null && _e$meta2 !== void 0 && _e$meta2.statusCode ? (_e$meta3 = e.meta) === null || _e$meta3 === void 0 ? void 0 : _e$meta3.statusCode : 500,
        body: {
          message: _i18n.i18n.translate('xpack.searchIndices.server.createIndex.errorMessage', {
            defaultMessage: 'Failed to create index due to an exception.\n{errorMessage}',
            values: {
              errorMessage: e.message
            }
          })
        }
      });
    }
  });
  router.post({
    path: _routes.SEARCH_DOCUMENTS_ROUTE,
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {
      body: _configSchema.schema.object({
        searchQuery: _configSchema.schema.string({
          defaultValue: ''
        }),
        trackTotalHits: _configSchema.schema.boolean({
          defaultValue: false
        })
      }),
      params: _configSchema.schema.object({
        indexName: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        page: _configSchema.schema.number({
          defaultValue: 0,
          min: 0
        }),
        size: _configSchema.schema.number({
          defaultValue: _types.DEFAULT_DOCS_PER_PAGE,
          min: 0
        })
      })
    }
  }, async (context, request, response) => {
    const client = (await context.core).elasticsearch.client.asCurrentUser;
    const indexName = decodeURIComponent(request.params.indexName);
    const searchQuery = request.body.searchQuery;
    const {
      page = 0,
      size = _types.DEFAULT_DOCS_PER_PAGE
    } = request.query;
    const from = page * size;
    const trackTotalHits = request.body.trackTotalHits;
    const searchResults = await (0, _lib.fetchSearchResults)(client, indexName, searchQuery, from, size, trackTotalHits);
    return response.ok({
      body: {
        results: searchResults
      },
      headers: {
        'content-type': 'application/json'
      }
    });
  });
}