"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.processRouter = exports.extractResponses = void 0;
var _coreHttpServer = require("@kbn/core-http-server");
var _coreHttpRouterServerInternal = require("@kbn/core-http-router-server-internal");
var _util = require("./util");
var _extract_authz_description = require("./extract_authz_description");
var _merge_operation = require("./merge_operation");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const processRouter = async ({
  appRouter,
  converter,
  getOpId,
  filters,
  env = {
    serverless: false
  }
}) => {
  const paths = {};
  if (filters !== null && filters !== void 0 && filters.version && filters.version !== _coreHttpRouterServerInternal.BASE_PUBLIC_VERSION) return {
    paths
  };
  const routes = (0, _util.prepareRoutes)(appRouter.getRoutes({
    excludeVersionedRoutes: true
  }), filters);
  for (const route of routes) {
    try {
      var _route$options, _route$options$descri, _route$options$summar;
      const pathParams = (0, _util.getPathParameters)(route.path);
      const validationSchemas = (0, _util.extractValidationSchemaFromRoute)(route);
      const contentType = (0, _util.extractContentType)((_route$options = route.options) === null || _route$options === void 0 ? void 0 : _route$options.body);
      const parameters = [...(0, _util.getXsrfHeaderForMethod)(route.method, route.options)];
      if (validationSchemas) {
        let pathObjects = [];
        let queryObjects = [];
        const reqParams = validationSchemas.params;
        if (reqParams) {
          pathObjects = converter.convertPathParameters(reqParams, pathParams);
        }
        const reqQuery = validationSchemas.query;
        if (reqQuery) {
          queryObjects = converter.convertQuery(reqQuery);
        }
        parameters.push(...pathObjects, ...queryObjects);
      }
      let description = `${(_route$options$descri = route.options.description) !== null && _route$options$descri !== void 0 ? _route$options$descri : ''}`;
      if (route.security) {
        const authzDescription = (0, _extract_authz_description.extractAuthzDescription)(route.security);
        description += `${route.options.description && authzDescription ? `<br/><br/>` : ''}${authzDescription !== null && authzDescription !== void 0 ? authzDescription : ''}`;
      }
      const hasDeprecations = !!route.options.deprecated;
      const operation = {
        summary: (_route$options$summar = route.options.summary) !== null && _route$options$summar !== void 0 ? _route$options$summar : '',
        tags: route.options.tags ? (0, _util.extractTags)(route.options.tags) : [],
        ...(description ? {
          description
        } : {}),
        ...(hasDeprecations ? {
          deprecated: true
        } : {}),
        ...(route.options.discontinued ? {
          'x-discontinued': route.options.discontinued
        } : {}),
        requestBody: !!(validationSchemas !== null && validationSchemas !== void 0 && validationSchemas.body) ? {
          content: {
            [(0, _util.getVersionedContentTypeString)(_coreHttpRouterServerInternal.BASE_PUBLIC_VERSION, 'public', contentType)]: {
              schema: converter.convert(validationSchemas.body)
            }
          }
        } : undefined,
        responses: extractResponses(route, converter),
        parameters,
        operationId: getOpId({
          path: route.path,
          method: route.method
        })
      };
      (0, _util.setXState)(route.options.availability, operation, env);
      if (route.options.oasOperationObject) {
        await (0, _merge_operation.mergeOperation)(route.options.oasOperationObject(), operation);
      }
      const path = {
        [route.method]: operation
      };
      (0, _util.assignToPaths)(paths, route.path, path);
    } catch (e) {
      // Enrich the error message with a bit more context
      e.message = `Error generating OpenAPI for route '${route.path}': ${e.message}`;
      throw e;
    }
  }
  return {
    paths
  };
};
exports.processRouter = processRouter;
const extractResponses = (route, converter) => {
  const responses = {};
  if (!route.validationSchemas) return responses;
  const fullConfig = (0, _coreHttpServer.getResponseValidation)(route.validationSchemas);
  if (fullConfig) {
    var _route$options2;
    const {
      unsafe,
      ...validationSchemas
    } = fullConfig;
    const contentType = (0, _util.extractContentType)((_route$options2 = route.options) === null || _route$options2 === void 0 ? void 0 : _route$options2.body);
    return Object.entries(validationSchemas).reduce((acc, [statusCode, schema]) => {
      var _acc$statusCode;
      const newContent = schema.body ? {
        [(0, _util.getVersionedContentTypeString)(_coreHttpRouterServerInternal.BASE_PUBLIC_VERSION, 'public', schema.bodyContentType ? [schema.bodyContentType] : contentType)]: {
          schema: converter.convert(schema.body())
        }
      } : undefined;
      acc[statusCode] = {
        ...acc[statusCode],
        description: schema.description,
        ...(0, _util.mergeResponseContent)(((_acc$statusCode = acc[statusCode]) !== null && _acc$statusCode !== void 0 ? _acc$statusCode : {}).content, newContent)
      };
      return acc;
    }, responses);
  }
  return responses;
};
exports.extractResponses = extractResponses;