"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.runDefaultAgentMode = void 0;
var _uuid = require("uuid");
var _rxjs = require("rxjs");
var _langchain = require("@kbn/onechat-genai-utils/langchain");
var _onechatCommon = require("@kbn/onechat-common");
var _utils = require("../utils");
var _capabilities = require("../utils/capabilities");
var _graph = require("./graph");
var _convert_graph_events = require("./convert_graph_events");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const chatAgentGraphName = 'default-onechat-agent';
/**
 * Create the handler function for the default onechat agent.
 */
const runDefaultAgentMode = async ({
  nextInput,
  conversation = [],
  capabilities,
  toolSelection = _onechatCommon.allToolsSelection,
  customInstructions,
  runId = (0, _uuid.v4)(),
  agentId,
  abortSignal
}, {
  logger,
  request,
  modelProvider,
  toolProvider,
  events
}) => {
  const model = await modelProvider.getDefaultModel();
  const resolvedCapabilities = (0, _capabilities.resolveCapabilities)(capabilities);
  logger.debug(`Running chat agent with connector: ${model.connector.name}, runId: ${runId}`);
  const selectedTools = await (0, _utils.selectProviderTools)({
    provider: toolProvider,
    selection: toolSelection,
    request
  });
  const manualEvents$ = new _rxjs.Subject();
  const eventEmitter = event => {
    manualEvents$.next(event);
  };
  const {
    tools: langchainTools,
    idMappings: toolIdMapping
  } = await (0, _langchain.toolsToLangchain)({
    tools: selectedTools,
    logger,
    request,
    sendEvent: eventEmitter
  });
  const initialMessages = (0, _utils.conversationToLangchainMessages)({
    nextInput,
    previousRounds: conversation
  });
  const agentGraph = (0, _graph.createAgentGraph)({
    logger,
    chatModel: model.chatModel,
    tools: langchainTools,
    capabilities: resolvedCapabilities,
    customInstructions
  });
  logger.debug(`Running chat agent with graph: ${chatAgentGraphName}, runId: ${runId}`);
  const eventStream = agentGraph.streamEvents({
    initialMessages
  }, {
    version: 'v2',
    signal: abortSignal,
    runName: chatAgentGraphName,
    metadata: {
      graphName: chatAgentGraphName,
      agentId,
      runId
    },
    recursionLimit: 25,
    callbacks: []
  });
  const graphEvents$ = (0, _rxjs.from)(eventStream).pipe((0, _rxjs.filter)(_langchain.isStreamEvent), (0, _convert_graph_events.convertGraphEvents)({
    graphName: chatAgentGraphName,
    toolIdMapping,
    logger
  }), (0, _rxjs.finalize)(() => manualEvents$.complete()));
  const events$ = (0, _rxjs.merge)(graphEvents$, manualEvents$).pipe((0, _utils.addRoundCompleteEvent)({
    userInput: nextInput
  }), (0, _rxjs.shareReplay)());
  events$.subscribe({
    next: event => events.emit(event),
    error: () => {
      // error will be handled by function return, we just need to trap here
    }
  });
  const round = await (0, _utils.extractRound)(events$);
  return {
    round
  };
};
exports.runDefaultAgentMode = runDefaultAgentMode;