"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deleteListItemRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _securitysolutionListConstants = require("@kbn/securitysolution-list-constants");
var _zodHelpers = require("@kbn/zod-helpers");
var _api = require("@kbn/securitysolution-lists-common/api");
var _utils = require("../utils");
var _ = require("..");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const deleteListItemRoute = router => {
  router.versioned.delete({
    access: 'public',
    path: _securitysolutionListConstants.LIST_ITEM_URL,
    security: {
      authz: {
        requiredPrivileges: ['lists-all']
      }
    }
  }).addVersion({
    validate: {
      request: {
        query: (0, _zodHelpers.buildRouteValidationWithZod)(_api.DeleteListItemRequestQuery)
      }
    },
    version: '2023-10-31'
  }, async (context, request, response) => {
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    try {
      const {
        id,
        list_id: listId,
        value,
        refresh
      } = request.query;
      const shouldRefresh = refresh === 'true' ? true : false;
      const lists = await (0, _.getListClient)(context);
      if (id != null) {
        const deleted = await lists.deleteListItem({
          id,
          refresh: shouldRefresh
        });
        if (deleted == null) {
          return siemResponse.error({
            body: `list item with id: "${id}" not found`,
            statusCode: 404
          });
        }
        return response.ok({
          body: _api.DeleteListItemResponse.parse(deleted)
        });
      } else if (listId != null && value != null) {
        const list = await lists.getList({
          id: listId
        });
        if (list == null) {
          return siemResponse.error({
            body: `list_id: "${listId}" does not exist`,
            statusCode: 404
          });
        }
        const deleted = await lists.deleteListItemByValue({
          listId,
          refresh: shouldRefresh,
          type: list.type,
          value
        });
        if (deleted == null || deleted.length === 0) {
          return siemResponse.error({
            body: `list_id: "${listId}" with ${value} was not found`,
            statusCode: 404
          });
        }
        return response.ok({
          body: _api.DeleteListItemResponse.parse(deleted)
        });
      } else {
        return siemResponse.error({
          body: 'Either "list_id" or "id" needs to be defined in the request',
          statusCode: 400
        });
      }
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return siemResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.deleteListItemRoute = deleteListItemRoute;