"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDefaultDefendInsightsGraph = void 0;
var _langgraph = require("@langchain/langgraph");
var _output_chunking = require("../../../langchain/output_chunking");
var _constants = require("../../../langchain/graphs/constants");
var _get_combined_prompt = require("./prompts/get_combined_prompt");
var _response_is_hallucinated = require("./helpers/response_is_hallucinated");
var _retriever = require("./nodes/retriever");
var _schemas = require("./schemas");
var _state = require("./state");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * This function returns a compiled state graph that represents the default
 * Defend Insights graph.
 */
const getDefaultDefendInsightsGraph = ({
  insightType,
  endpointIds,
  anonymizationFields,
  esClient,
  kbDataClient,
  llm,
  logger,
  onNewReplacements,
  prompts,
  replacements,
  size,
  start,
  end
}) => {
  try {
    const graphState = (0, _state.getDefaultGraphState)({
      prompts,
      start,
      end
    });

    // get nodes:
    const retrieveAnonymizedEventsNode = (0, _retriever.getRetrieveAnonymizedEventsNode)({
      insightType,
      endpointIds,
      anonymizationFields,
      esClient,
      kbDataClient,
      logger,
      onNewReplacements,
      replacements,
      size
    });
    const generationSchema = (0, _schemas.getDefendInsightsSchema)({
      type: insightType,
      prompts
    });
    const generateNode = (0, _output_chunking.getGenerateNode)({
      llm,
      logger,
      getCombinedPromptFn: _get_combined_prompt.getCombinedDefendInsightsPrompt,
      responseIsHallucinated: _response_is_hallucinated.responseIsHallucinated,
      generationSchema
    });
    const refineNode = (0, _output_chunking.getRefineNode)({
      llm,
      logger,
      responseIsHallucinated: _response_is_hallucinated.responseIsHallucinated,
      generationSchema
    });

    // get edges:
    const generateOrEndEdge = (0, _output_chunking.getGenerateOrEndEdge)(logger);
    const generateOrRefineOrEndEdge = (0, _output_chunking.getGenerateOrRefineOrEndEdge)(logger);
    const refineOrEndEdge = (0, _output_chunking.getRefineOrEndEdge)(logger);
    const retrieveAnonymizedEventsOrGenerateEdge = (0, _output_chunking.getRetrieveAnonymizedDocsOrGenerateEdge)(logger);

    // create the graph:
    const graph = new _langgraph.StateGraph({
      channels: graphState
    }).addNode(_constants.NodeType.RETRIEVE_ANONYMIZED_DOCS_NODE, retrieveAnonymizedEventsNode).addNode(_constants.NodeType.GENERATE_NODE, generateNode).addNode(_constants.NodeType.REFINE_NODE, refineNode).addConditionalEdges(_langgraph.START, retrieveAnonymizedEventsOrGenerateEdge, {
      generate: _constants.NodeType.GENERATE_NODE,
      retrieve_anonymized_docs: _constants.NodeType.RETRIEVE_ANONYMIZED_DOCS_NODE
    }).addConditionalEdges(_constants.NodeType.RETRIEVE_ANONYMIZED_DOCS_NODE, generateOrEndEdge, {
      end: _langgraph.END,
      generate: _constants.NodeType.GENERATE_NODE
    }).addConditionalEdges(_constants.NodeType.GENERATE_NODE, generateOrRefineOrEndEdge, {
      end: _langgraph.END,
      generate: _constants.NodeType.GENERATE_NODE,
      refine: _constants.NodeType.REFINE_NODE
    }).addConditionalEdges(_constants.NodeType.REFINE_NODE, refineOrEndEdge, {
      end: _langgraph.END,
      refine: _constants.NodeType.REFINE_NODE
    });
    return graph.compile();
  } catch (e) {
    throw new Error(`Unable to compile DefendInsightsGraph\n${e}`);
  }
};
exports.getDefaultDefendInsightsGraph = getDefaultDefendInsightsGraph;