"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getFlattenedKeys = getFlattenedKeys;
exports.pickValuesBasedOnStructure = pickValuesBasedOnStructure;
var _lodash = require("lodash");
var _saferLodashSet = require("@kbn/safer-lodash-set");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function getFlattenedKeys(obj) {
  const result = [];
  const keys = Object.keys(obj).map(k => `["${k}"]`);
  let current;
  while (keys.length) {
    const key = keys.pop();
    current = (0, _lodash.get)(obj, key);
    if ((0, _lodash.isPlainObject)(current)) {
      const innerKeys = Object.keys(current);
      if (innerKeys.length < 1) {
        result.push(`${key}`);
      } else {
        for (const innerKey of innerKeys) {
          keys.unshift(`${key}["${innerKey}"]`);
        }
      }
    } else if (Array.isArray(current)) {
      const arr = current;
      if (arr.length < 1) {
        result.push(key);
      } else {
        for (let i = 0; i < arr.length; i++) keys.unshift(`${key}[${i}]`);
      }
    } else {
      result.push(key);
    }
  }
  return result;
}

/**
 * Given two objects, use the first object as a structural map to extract values
 * from a second object, preserving the placement in the first object.
 *
 * @example
 * ```ts
 * const keySource = { a: 1, b: [{ a: 1 }, { a: 2 }] };
 * const target = { a: 2, b: [{ a: 2, b: 3 }, { a: 3, b: 4 }] };
 * pickValuesBasedOnStructure(keySource, target);
 * // => { a: 2, b: [{ a: 2 }, { a: 3 }] }
 * ```
 *
 * @note This is intended to specifically be used in the application of forward
 *       compatibility schemas when loading a saved object from the database,
 *       downgrading it and keeping only the known, validated subset of values.
 */
function pickValuesBasedOnStructure(structuralSource, target) {
  const paths = getFlattenedKeys(structuralSource);
  const result = {};
  for (const path of paths) {
    if (!(0, _lodash.has)(target, path)) continue;
    const value = (0, _lodash.get)(target, path);
    if (Array.isArray(value)) {
      (0, _saferLodashSet.set)(result, path, []);
    } else if ((0, _lodash.isPlainObject)(value)) {
      (0, _saferLodashSet.set)(result, path, {});
    } else {
      (0, _saferLodashSet.set)(result, path, value);
    }
  }
  return result;
}