"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useCreateToolService = exports.useCreateTool = void 0;
var _onechatBrowser = require("@kbn/onechat-browser");
var _reactQuery = require("@tanstack/react-query");
var _immer = require("immer");
var _react = require("react");
var _query_keys = require("../../query_keys");
var _duplicate_name = require("../../utils/duplicate_name");
var _i18n = require("../../utils/i18n");
var _use_onechat_service = require("../use_onechat_service");
var _use_toasts = require("../use_toasts");
var _use_tools = require("./use_tools");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useCreateToolService = ({
  onSuccess,
  onError
} = {}) => {
  const queryClient = (0, _reactQuery.useQueryClient)();
  const {
    toolsService
  } = (0, _use_onechat_service.useOnechatServices)();
  const {
    mutate,
    mutateAsync,
    isLoading
  } = (0, _reactQuery.useMutation)({
    mutationFn: tool => toolsService.create(tool),
    onSuccess,
    onError,
    onSettled: () => queryClient.invalidateQueries({
      queryKey: _query_keys.queryKeys.tools.all
    })
  });
  return {
    createToolSync: mutate,
    createTool: mutateAsync,
    isLoading
  };
};
exports.useCreateToolService = useCreateToolService;
const useCreateTool = ({
  sourceToolId,
  onSuccess,
  onError,
  onLoadingError
} = {}) => {
  const {
    addSuccessToast,
    addErrorToast
  } = (0, _use_toasts.useToasts)();
  const {
    tool: sourceTool,
    isLoading
  } = (0, _use_tools.useTool)({
    toolId: sourceToolId,
    onLoadingError
  });
  const handleSuccess = (0, _react.useCallback)((response, variables, context) => {
    addSuccessToast({
      title: _i18n.labels.tools.createEsqlToolSuccessToast(response.id)
    });
    onSuccess === null || onSuccess === void 0 ? void 0 : onSuccess(response, variables, context);
  }, [addSuccessToast, onSuccess]);
  const handleError = (0, _react.useCallback)((error, variables, context) => {
    addErrorToast({
      title: _i18n.labels.tools.createEsqlToolErrorToast,
      text: (0, _onechatBrowser.formatOnechatErrorMessage)(error)
    });
    onError === null || onError === void 0 ? void 0 : onError(error, variables, context);
  }, [addErrorToast, onError]);
  const {
    createTool,
    isLoading: isSubmitting
  } = useCreateToolService({
    onSuccess: handleSuccess,
    onError: handleError
  });
  const handleCreateTool = (0, _react.useCallback)(async tool => {
    return createTool(tool);
  }, [createTool]);
  const sourceToolClone = (0, _react.useMemo)(() => {
    if (!sourceTool) {
      return;
    }
    return (0, _immer.produce)(sourceTool, draft => {
      draft.id = (0, _duplicate_name.duplicateName)(sourceTool.id);
    });
  }, [sourceTool]);
  return {
    sourceTool: sourceToolClone,
    isLoading: !!sourceToolId && isLoading,
    isSubmitting,
    createTool: handleCreateTool
  };
};
exports.useCreateTool = useCreateTool;