"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchAggIntervals = void 0;
var _lodash = require("lodash");
var _fieldTypes = require("@kbn/field-types");
var _mlIsPopulatedObject = require("@kbn/ml-is-populated-object");
var _mlStringHash = require("@kbn/ml-string-hash");
var _mlRandomSamplerUtils = require("@kbn/ml-random-sampler-utils");
var _build_sampler_aggregation = require("./build_sampler_aggregation");
var _get_sampler_aggregations_response_path = require("./get_sampler_aggregations_response_path");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_CHART_COLUMNS = 20;

/**
 * Interface for the parameters required to fetch aggregation intervals.
 */

/**
 * Asynchronously fetches aggregation intervals from an Elasticsearch client.
 *
 * @param params - The parameters for fetching aggregation intervals.
 * @returns A promise that resolves to a map of numeric column statistics.
 */
const fetchAggIntervals = async params => {
  const {
    esClient,
    abortSignal,
    arguments: args
  } = params;
  const {
    indexPattern,
    query,
    fields,
    samplerShardSize,
    runtimeMappings,
    randomSamplerProbability,
    randomSamplerSeed
  } = args;
  if (samplerShardSize >= 1 && randomSamplerProbability !== undefined && randomSamplerProbability < 1) {
    throw new Error('Sampler and Random Sampler cannot be used at the same time.');
  }
  const numericColumns = fields.filter(field => {
    return field.type === _fieldTypes.KBN_FIELD_TYPES.NUMBER || field.type === _fieldTypes.KBN_FIELD_TYPES.DATE;
  });
  if (numericColumns.length === 0) {
    return {};
  }
  const minMaxAggs = numericColumns.reduce((aggs, c) => {
    const id = (0, _mlStringHash.stringHash)(c.fieldName);
    aggs[id] = {
      stats: {
        field: c.fieldName
      }
    };
    return aggs;
  }, {});
  const {
    wrap,
    unwrap
  } = (0, _mlRandomSamplerUtils.createRandomSamplerWrapper)({
    probability: randomSamplerProbability !== null && randomSamplerProbability !== void 0 ? randomSamplerProbability : 1,
    seed: randomSamplerSeed
  });
  const body = await esClient.search({
    index: indexPattern,
    size: 0,
    query,
    aggs: randomSamplerProbability === undefined ? (0, _build_sampler_aggregation.buildSamplerAggregation)(minMaxAggs, samplerShardSize) : wrap(minMaxAggs),
    ...((0, _mlIsPopulatedObject.isPopulatedObject)(runtimeMappings) ? {
      runtime_mappings: runtimeMappings
    } : {})
  }, {
    signal: abortSignal,
    maxRetries: 0
  });
  const aggsPath = randomSamplerProbability === undefined ? (0, _get_sampler_aggregations_response_path.getSamplerAggregationsResponsePath)(samplerShardSize) : [];
  const aggregations = aggsPath.length > 0 ? (0, _lodash.get)(body.aggregations, aggsPath) : randomSamplerProbability !== undefined && body.aggregations !== undefined ? unwrap(body.aggregations) : body.aggregations;
  return Object.keys(aggregations).reduce((p, aggName) => {
    if (aggregations === undefined) {
      return p;
    }
    const stats = [aggregations[aggName].min, aggregations[aggName].max];
    if (!stats.includes(null)) {
      const delta = aggregations[aggName].max - aggregations[aggName].min;
      let aggInterval = 1;
      if (delta > MAX_CHART_COLUMNS || delta <= 1) {
        aggInterval = delta / (MAX_CHART_COLUMNS - 1);
      }
      p[aggName] = {
        interval: aggInterval,
        min: stats[0],
        max: stats[1]
      };
    }
    return p;
  }, {});
};
exports.fetchAggIntervals = fetchAggIntervals;