"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useDatasetFiltering = void 0;
var _react = require("react");
var _public = require("@kbn/ml-plugin/public");
var _use_kibana = require("../../../hooks/use_kibana");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const initialState = {
  selectedDatasets: [],
  selectedDatasetsFilters: []
};
function reducer(state, action) {
  switch (action.type) {
    case 'changeSelectedDatasets':
      return {
        ...state,
        selectedDatasets: action.payload.datasets
      };
    case 'updateDatasetsFilters':
      const datasetsToAdd = action.payload.filters.reduce((prevFilters, nextFilter) => {
        var _nextFilter$meta$para;
        const query = typeof nextFilter.meta.params === 'object' && 'query' in nextFilter.meta.params && ((_nextFilter$meta$para = nextFilter.meta.params) === null || _nextFilter$meta$para === void 0 ? void 0 : _nextFilter$meta$para.query);
        const queryString = query ? String(query) : '';
        if (!state.selectedDatasets.includes(queryString)) {
          prevFilters.push(queryString);
        }
        return prevFilters;
      }, []);
      return {
        ...state,
        selectedDatasets: [...state.selectedDatasets, ...datasetsToAdd],
        selectedDatasetsFilters: action.payload.filters
      };
    default:
      throw new Error('Unknown action');
  }
}
const useDatasetFiltering = () => {
  const {
    services
  } = (0, _use_kibana.useKibanaContextForPlugin)();
  const [reducerState, dispatch] = (0, _react.useReducer)(reducer, initialState);
  const handleSetSelectedDatasets = (0, _react.useCallback)(datasets => {
    dispatch({
      type: 'changeSelectedDatasets',
      payload: {
        datasets
      }
    });
  }, [dispatch]);

  // NOTE: The anomaly swimlane embeddable will communicate it's filter action
  // changes via the filterManager service.
  (0, _react.useEffect)(() => {
    const sub = services.data.query.filterManager.getUpdates$().subscribe(() => {
      const filters = services.data.query.filterManager.getFilters().filter(filter => filter.meta.controlledBy && filter.meta.controlledBy === _public.CONTROLLED_BY_SWIM_LANE_FILTER);
      dispatch({
        type: 'updateDatasetsFilters',
        payload: {
          filters
        }
      });
    });
    return () => sub.unsubscribe();
  }, [services.data.query.filterManager, dispatch]);

  // NOTE: When filters are removed via the UI we need to make sure these are also tidied up
  // within the FilterManager service, otherwise a scenario can occur where that filter can't
  // be re-added via the embeddable as it will be seen as a duplicate to the FilterManager,
  // and no update will be emitted.
  (0, _react.useEffect)(() => {
    const filtersToRemove = reducerState.selectedDatasetsFilters.filter(filter => {
      var _filter$meta$params;
      const query = typeof filter.meta.params === 'object' && 'query' in filter.meta.params && ((_filter$meta$params = filter.meta.params) === null || _filter$meta$params === void 0 ? void 0 : _filter$meta$params.query);
      const queryString = query ? String(query) : '';
      return !reducerState.selectedDatasets.includes(queryString);
    });
    if (filtersToRemove.length > 0) {
      filtersToRemove.forEach(filter => {
        services.data.query.filterManager.removeFilter(filter);
      });
    }
  }, [reducerState.selectedDatasets, reducerState.selectedDatasetsFilters, services.data.query.filterManager]);
  return {
    selectedDatasets: reducerState.selectedDatasets,
    setSelectedDatasets: handleSetSelectedDatasets,
    selectedDatasetsFilters: reducerState.selectedDatasetsFilters
  };
};
exports.useDatasetFiltering = useDatasetFiltering;