"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerRoutes = exports.READ_PACKAGE_INFO_SECURITY = exports.READ_PACKAGE_INFO_AUTHZ = exports.INSTALL_PACKAGES_SECURITY = exports.INSTALL_PACKAGES_AUTHZ = void 0;
var _experimental_features = require("../../../common/experimental_features");
var _constants = require("../../../common/constants");
var _security = require("../../services/security");
var _constants2 = require("../../constants");
var _types = require("../../types");
var _api_privileges = require("../../constants/api_privileges");
var _errors = require("../schema/errors");
var _handlers = require("./handlers");
var _file_handler = require("./file_handler");
var _install_assets_handler = require("./install_assets_handler");
var _bulk_handler = require("./bulk_handler");
var _package_datastream_assets_handler = require("./package_datastream_assets_handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_FILE_SIZE_BYTES = 104857600; // 100MB

const INSTALL_PACKAGES_AUTHZ = exports.INSTALL_PACKAGES_AUTHZ = {
  integrations: {
    installPackages: true
  }
};
const INSTALL_PACKAGES_SECURITY = exports.INSTALL_PACKAGES_SECURITY = {
  authz: {
    requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.INTEGRATIONS.ALL, _api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.ALL]
  }
};
const READ_PACKAGE_INFO_AUTHZ = exports.READ_PACKAGE_INFO_AUTHZ = {
  integrations: {
    readPackageInfo: true
  }
};
const READ_PACKAGE_INFO_SECURITY = exports.READ_PACKAGE_INFO_SECURITY = {
  authz: {
    requiredPrivileges: [{
      anyRequired: [_api_privileges.FLEET_API_PRIVILEGES.INTEGRATIONS.READ, _api_privileges.FLEET_API_PRIVILEGES.SETUP, _api_privileges.FLEET_API_PRIVILEGES.FLEET.ALL]
    }]
  }
};
const registerRoutes = (router, config) => {
  const experimentalFeatures = (0, _experimental_features.parseExperimentalConfigValue)(config.enableExperimental);
  router.versioned.get({
    path: _constants2.EPM_API_ROUTES.CATEGORIES_PATTERN,
    security: READ_PACKAGE_INFO_SECURITY,
    summary: `Get package categories`,
    options: {
      tags: ['oas-tag:Elastic Package Manager (EPM)']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetCategoriesRequestSchema,
      response: {
        200: {
          body: () => _types.GetCategoriesResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.getCategoriesHandler);
  router.versioned.get({
    path: _constants2.EPM_API_ROUTES.LIST_PATTERN,
    security: READ_PACKAGE_INFO_SECURITY,
    summary: `Get packages`,
    options: {
      tags: ['oas-tag:Elastic Package Manager (EPM)']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetPackagesRequestSchema,
      response: {
        200: {
          body: () => _types.GetPackagesResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.getListHandler);
  router.versioned.get({
    path: _constants2.EPM_API_ROUTES.INSTALLED_LIST_PATTERN,
    security: READ_PACKAGE_INFO_SECURITY,
    summary: `Get installed packages`,
    options: {
      tags: ['oas-tag:Elastic Package Manager (EPM)']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetInstalledPackagesRequestSchema,
      response: {
        200: {
          body: () => _types.GetInstalledPackagesResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.getInstalledListHandler);
  router.versioned.get({
    path: _constants2.EPM_API_ROUTES.LIMITED_LIST_PATTERN,
    security: READ_PACKAGE_INFO_SECURITY,
    summary: `Get a limited package list`,
    options: {
      tags: ['oas-tag:Elastic Package Manager (EPM)']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: {},
      response: {
        200: {
          body: () => _types.GetLimitedPackagesResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.getLimitedListHandler);
  router.versioned.get({
    path: _constants2.EPM_API_ROUTES.STATS_PATTERN,
    security: READ_PACKAGE_INFO_SECURITY,
    summary: `Get package stats`,
    options: {
      tags: ['oas-tag:Elastic Package Manager (EPM)']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetStatsRequestSchema,
      response: {
        200: {
          body: () => _types.GetStatsResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.getStatsHandler);
  router.versioned.get({
    path: _constants2.EPM_API_ROUTES.INPUTS_PATTERN,
    security: READ_PACKAGE_INFO_SECURITY,
    summary: `Get an inputs template`,
    options: {
      tags: ['oas-tag:Elastic Package Manager (EPM)']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetInputsRequestSchema,
      response: {
        200: {
          body: () => _types.GetInputsResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.getInputsHandler);
  router.versioned.get({
    path: _constants2.EPM_API_ROUTES.FILEPATH_PATTERN,
    security: READ_PACKAGE_INFO_SECURITY,
    summary: `Get a package file`,
    options: {
      tags: ['oas-tag:Elastic Package Manager (EPM)']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetFileRequestSchema,
      response: {
        200: {
          body: () => _types.GetFileResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _file_handler.getFileHandler);
  router.versioned
  // @ts-ignore TODO move to kibana authz https://github.com/elastic/kibana/issues/203170
  .get({
    path: _constants2.EPM_API_ROUTES.INFO_PATTERN,
    fleetAuthz: fleetAuthz => (0, _security.calculateRouteAuthz)(fleetAuthz, (0, _security.getRouteRequiredAuthz)('get', _constants2.EPM_API_ROUTES.INFO_PATTERN)).granted,
    summary: `Get a package`,
    options: {
      tags: ['oas-tag:Elastic Package Manager (EPM)']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetInfoRequestSchema,
      response: {
        200: {
          body: () => _types.GetInfoResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.getInfoHandler);
  router.versioned.get({
    path: _constants2.EPM_API_ROUTES.KNOWLEDGE_BASE_PATTERN,
    fleetAuthz: fleetAuthz => (0, _security.calculateRouteAuthz)(fleetAuthz, (0, _security.getRouteRequiredAuthz)('get', _constants2.EPM_API_ROUTES.KNOWLEDGE_BASE_PATTERN)).granted,
    security: {
      authz: {
        enabled: false,
        reason: 'This route uses Fleet authorization via fleetAuthz instead of standard Kibana authorization'
      }
    },
    summary: `Get all knowledge base content for a package`,
    options: {
      tags: ['internal', 'oas-tag:Elastic Package Manager (EPM)']
    },
    access: 'internal'
  }).addVersion({
    version: _constants.API_VERSIONS.internal.v1,
    validate: {
      request: _types.GetKnowledgeBaseRequestSchema,
      response: {
        200: {
          body: () => _types.GetKnowledgeBaseResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.getKnowledgeBaseHandler);
  router.versioned.put({
    path: _constants2.EPM_API_ROUTES.INFO_PATTERN,
    security: INSTALL_PACKAGES_SECURITY,
    summary: `Update package settings`,
    options: {
      tags: ['oas-tag:Elastic Package Manager (EPM)']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.UpdatePackageRequestSchema,
      response: {
        200: {
          body: () => _types.UpdatePackageResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.updatePackageHandler);
  router.versioned.post({
    path: _constants2.EPM_API_ROUTES.INSTALL_FROM_REGISTRY_PATTERN,
    security: INSTALL_PACKAGES_SECURITY,
    summary: `Install a package from the registry`,
    options: {
      tags: ['oas-tag:Elastic Package Manager (EPM)']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.InstallPackageFromRegistryRequestSchema,
      response: {
        200: {
          body: () => _types.InstallPackageResponseSchema,
          description: 'OK'
        },
        400: {
          body: _errors.genericErrorResponse,
          description: 'Bad Request'
        }
      }
    }
  }, _handlers.installPackageFromRegistryHandler);
  router.versioned.post({
    path: _constants2.EPM_API_ROUTES.INSTALL_RULE_ASSETS_PATTERN,
    security: INSTALL_PACKAGES_SECURITY,
    summary: `Install Kibana alert rule for a package`,
    options: {
      tags: ['oas-tag:Elastic Package Manager (EPM)']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.InstallRuleAssetsRequestSchema,
      response: {
        200: {
          body: () => _types.InstallKibanaAssetsResponseSchema,
          description: 'OK'
        },
        400: {
          body: _errors.genericErrorResponse,
          description: 'Bad Request'
        }
      }
    }
  }, _install_assets_handler.installRuleAssetsHandler);
  router.versioned.post({
    path: _constants2.EPM_API_ROUTES.INSTALL_KIBANA_ASSETS_PATTERN,
    security: INSTALL_PACKAGES_SECURITY,
    summary: `Install Kibana assets for a package`,
    options: {
      tags: ['oas-tag:Elastic Package Manager (EPM)']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.InstallKibanaAssetsRequestSchema,
      response: {
        200: {
          body: () => _types.InstallKibanaAssetsResponseSchema,
          description: 'OK'
        },
        400: {
          body: _errors.genericErrorResponse,
          description: 'Bad Request'
        }
      }
    }
  }, _install_assets_handler.installPackageKibanaAssetsHandler);
  router.versioned.delete({
    path: _constants2.EPM_API_ROUTES.DELETE_KIBANA_ASSETS_PATTERN,
    security: INSTALL_PACKAGES_SECURITY,
    summary: `Delete Kibana assets for a package`,
    options: {
      tags: ['oas-tag:Elastic Package Manager (EPM)']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.DeleteKibanaAssetsRequestSchema,
      response: {
        200: {
          body: () => _types.InstallKibanaAssetsResponseSchema,
          description: 'OK'
        },
        400: {
          body: _errors.genericErrorResponse,
          description: 'Bad Request'
        }
      }
    }
  }, _install_assets_handler.deletePackageKibanaAssetsHandler);
  if (experimentalFeatures.installedIntegrationsTabularUI) {
    router.versioned.post({
      path: _constants2.EPM_API_ROUTES.BULK_UPGRADE_PATTERN,
      security: INSTALL_PACKAGES_SECURITY,
      summary: `Bulk upgrade packages`,
      options: {
        tags: ['oas-tag:Elastic Package Manager (EPM)']
      }
    }).addVersion({
      version: _constants.API_VERSIONS.public.v1,
      validate: {
        request: _types.BulkUpgradePackagesRequestSchema,
        response: {
          200: {
            body: () => _types.BulkUpgradePackagesResponseSchema,
            description: 'OK'
          },
          400: {
            body: _errors.genericErrorResponse,
            description: 'Bad Request'
          }
        }
      }
    }, _bulk_handler.postBulkUpgradePackagesHandler);
    router.versioned.post({
      path: _constants2.EPM_API_ROUTES.BULK_UNINSTALL_PATTERN,
      security: INSTALL_PACKAGES_SECURITY,
      summary: `Bulk uninstall packages`,
      options: {
        tags: ['oas-tag:Elastic Package Manager (EPM)']
      }
    }).addVersion({
      version: _constants.API_VERSIONS.public.v1,
      validate: {
        request: _types.BulkUninstallPackagesRequestSchema,
        response: {
          200: {
            body: () => _types.BulkUpgradePackagesResponseSchema,
            description: 'OK'
          },
          400: {
            body: _errors.genericErrorResponse,
            description: 'Bad Request'
          }
        }
      }
    }, _bulk_handler.postBulkUninstallPackagesHandler);
    if (experimentalFeatures.enablePackageRollback) {
      router.versioned.post({
        path: _constants2.EPM_API_ROUTES.BULK_ROLLBACK_PATTERN,
        security: INSTALL_PACKAGES_SECURITY,
        summary: `Bulk rollback packages`,
        options: {
          tags: ['oas-tag:Elastic Package Manager (EPM)']
        }
      }).addVersion({
        version: _constants.API_VERSIONS.public.v1,
        validate: {
          request: _types.BulkRollbackPackagesRequestSchema,
          response: {
            200: {
              body: () => _types.BulkRollbackPackagesResponseSchema,
              description: 'OK'
            },
            400: {
              body: _errors.genericErrorResponse,
              description: 'Bad Request'
            }
          }
        }
      }, _bulk_handler.postBulkRollbackPackagesHandler);
      router.versioned.get({
        path: _constants2.EPM_API_ROUTES.BULK_ROLLBACK_INFO_PATTERN,
        security: INSTALL_PACKAGES_SECURITY,
        summary: `Get Bulk rollback packages details`,
        options: {
          tags: ['oas-tag:Elastic Package Manager (EPM)']
        }
      }).addVersion({
        version: _constants.API_VERSIONS.public.v1,
        validate: {
          request: _types.GetOneBulkOperationPackagesRequestSchema,
          response: {
            200: {
              body: () => _types.GetOneBulkOperationPackagesResponseSchema,
              description: 'OK'
            },
            400: {
              body: _errors.genericErrorResponse,
              description: 'Bad Request'
            }
          }
        }
      }, _bulk_handler.getOneBulkOperationPackagesHandler);
    }
    router.versioned.get({
      path: _constants2.EPM_API_ROUTES.BULK_UNINSTALL_INFO_PATTERN,
      security: INSTALL_PACKAGES_SECURITY,
      summary: `Get Bulk uninstall packages details`,
      options: {
        tags: ['oas-tag:Elastic Package Manager (EPM)']
      }
    }).addVersion({
      version: _constants.API_VERSIONS.public.v1,
      validate: {
        request: _types.GetOneBulkOperationPackagesRequestSchema,
        response: {
          200: {
            body: () => _types.GetOneBulkOperationPackagesResponseSchema,
            description: 'OK'
          },
          400: {
            body: _errors.genericErrorResponse,
            description: 'Bad Request'
          }
        }
      }
    }, _bulk_handler.getOneBulkOperationPackagesHandler);
    router.versioned.get({
      path: _constants2.EPM_API_ROUTES.BULK_UPGRADE_INFO_PATTERN,
      security: INSTALL_PACKAGES_SECURITY,
      summary: `Get Bulk upgrade packages details`,
      options: {
        tags: ['oas-tag:Elastic Package Manager (EPM)']
      }
    }).addVersion({
      version: _constants.API_VERSIONS.public.v1,
      validate: {
        request: _types.GetOneBulkOperationPackagesRequestSchema,
        response: {
          200: {
            body: () => _types.GetOneBulkOperationPackagesResponseSchema,
            description: 'OK'
          },
          400: {
            body: _errors.genericErrorResponse,
            description: 'Bad Request'
          }
        }
      }
    }, _bulk_handler.getOneBulkOperationPackagesHandler);
  }
  router.versioned.post({
    path: _constants2.EPM_API_ROUTES.BULK_INSTALL_PATTERN,
    security: INSTALL_PACKAGES_SECURITY,
    summary: `Bulk install packages`,
    options: {
      tags: ['oas-tag:Elastic Package Manager (EPM)']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.BulkInstallPackagesFromRegistryRequestSchema,
      response: {
        200: {
          body: () => _types.BulkInstallPackagesFromRegistryResponseSchema,
          description: 'OK'
        },
        400: {
          body: _errors.genericErrorResponse,
          description: 'Bad Request'
        }
      }
    }
  }, _handlers.bulkInstallPackagesFromRegistryHandler);

  // Only allow upload for superuser
  router.versioned.post({
    path: _constants2.EPM_API_ROUTES.INSTALL_BY_UPLOAD_PATTERN,
    options: {
      body: {
        accepts: ['application/gzip', 'application/zip'],
        parse: false,
        maxBytes: MAX_FILE_SIZE_BYTES
      },
      tags: [`oas-tag:Elastic Package Manager (EPM)`]
    },
    security: INSTALL_PACKAGES_SECURITY,
    summary: `Install a package by upload`
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.InstallPackageByUploadRequestSchema,
      response: {
        200: {
          body: () => _types.InstallPackageResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.installPackageByUploadHandler);
  router.versioned.post({
    path: _constants2.EPM_API_ROUTES.CUSTOM_INTEGRATIONS_PATTERN,
    security: INSTALL_PACKAGES_SECURITY,
    summary: `Create a custom integration`,
    options: {
      tags: ['oas-tag:Elastic Package Manager (EPM)']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.CreateCustomIntegrationRequestSchema,
      response: {
        200: {
          body: () => _types.InstallPackageResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.createCustomIntegrationHandler);
  router.versioned.delete({
    path: _constants2.EPM_API_ROUTES.DELETE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.INTEGRATIONS.ALL, _api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.ALL]
      }
    },
    summary: `Delete a package`,
    options: {
      tags: ['oas-tag:Elastic Package Manager (EPM)']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.DeletePackageRequestSchema,
      response: {
        200: {
          body: () => _types.DeletePackageResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.deletePackageHandler);
  router.versioned.get({
    path: _constants2.EPM_API_ROUTES.VERIFICATION_KEY_ID,
    security: READ_PACKAGE_INFO_SECURITY,
    summary: `Get a package signature verification key ID`,
    options: {
      tags: ['oas-tag:Elastic Package Manager (EPM)']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: {},
      response: {
        200: {
          body: () => _types.GetVerificationKeyIdResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.getVerificationKeyIdHandler);
  router.versioned.get({
    path: _constants2.EPM_API_ROUTES.DATA_STREAMS_PATTERN,
    security: READ_PACKAGE_INFO_SECURITY,
    summary: `Get data streams`,
    options: {
      tags: ['oas-tag:Data streams']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetDataStreamsRequestSchema,
      response: {
        200: {
          body: () => _types.GetDataStreamsResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.getDataStreamsHandler);
  router.versioned.post({
    path: _constants2.EPM_API_ROUTES.BULK_ASSETS_PATTERN,
    security: READ_PACKAGE_INFO_SECURITY,
    summary: `Bulk get assets`,
    options: {
      tags: ['oas-tag:Elastic Package Manager (EPM)']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetBulkAssetsRequestSchema,
      response: {
        200: {
          body: () => _types.GetBulkAssetsResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.getBulkAssetsHandler);

  // Update transforms with es-secondary-authorization headers,
  // append authorized_by to transform's _meta, and start transforms
  router.versioned
  // @ts-ignore TODO move to kibana authz https://github.com/elastic/kibana/issues/203170
  .post({
    path: _constants2.EPM_API_ROUTES.REAUTHORIZE_TRANSFORMS,
    fleetAuthz: {
      ...INSTALL_PACKAGES_AUTHZ,
      packagePrivileges: {
        transform: {
          actions: {
            canStartStopTransform: {
              executePackageAction: true
            }
          }
        }
      }
    },
    summary: `Authorize transforms`,
    options: {
      tags: ['oas-tag:Elastic Package Manager (EPM)']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.ReauthorizeTransformRequestSchema,
      response: {
        200: {
          body: () => _types.ReauthorizeTransformResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.reauthorizeTransformsHandler);
  router.versioned.put({
    path: _constants2.EPM_API_ROUTES.UPDATE_CUSTOM_INTEGRATIONS_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.SETTINGS.ALL, _api_privileges.FLEET_API_PRIVILEGES.INTEGRATIONS.ALL]
      }
    },
    summary: `Update a custom integration`,
    options: {
      tags: ['oas-tag:Elastic Package Manager (EPM)']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.CustomIntegrationRequestSchema,
      response: {
        200: {},
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.updateCustomIntegrationHandler);
  router.versioned.delete({
    path: _constants2.EPM_API_ROUTES.PACKAGES_DATASTREAM_ASSETS,
    security: INSTALL_PACKAGES_SECURITY,
    summary: `Delete assets for an input package`,
    options: {
      tags: ['oas-tag:Elastic Package Manager (EPM)']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.DeletePackageDatastreamAssetsRequestSchema,
      response: {
        200: {
          body: () => _types.DeletePackageDatastreamAssetsResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _package_datastream_assets_handler.deletePackageDatastreamAssetsHandler);
  if (experimentalFeatures.enablePackageRollback) {
    router.versioned.post({
      path: _constants2.EPM_API_ROUTES.ROLLBACK_PATTERN,
      security: INSTALL_PACKAGES_SECURITY,
      summary: `Rollback a package to previous version`,
      options: {
        tags: ['oas-tag:Elastic Package Manager (EPM)'],
        availability: {
          since: '9.1.0',
          stability: 'experimental'
        }
      }
    }).addVersion({
      version: _constants.API_VERSIONS.public.v1,
      validate: {
        request: _types.RollbackPackageRequestSchema,
        response: {
          200: {
            body: () => _types.RollbackPackageResponseSchema
          },
          400: {
            body: _errors.genericErrorResponse
          }
        }
      }
    }, _handlers.rollbackPackageHandler);
  }
};
exports.registerRoutes = registerRoutes;