"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.setWithParameter = exports.setTargetField = exports.setQuery = exports.setFields = exports.list = void 0;
var _builder = require("../../../builder");
var generic = _interopRequireWildcard(require("../../generic"));
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Lists all "RERANK" commands in the query AST.
 *
 * @param ast The root AST node to search for "RERANK" commands.
 * @returns A collection of "RERANK" commands.
 */
const list = ast => {
  return generic.commands.list(ast, cmd => cmd.name === 'rerank');
};

/**
 * Sets or updates the query text for the RERANK command.
 *
 * @param cmd The RERANK command AST node to modify.
 * @param query The query text to set.
 */
exports.list = list;
const setQuery = (cmd, query) => {
  const queryLiteral = typeof query === 'string' ? _builder.Builder.expression.literal.string(query) : query;
  const firstArg = cmd.args[0];
  cmd.query = queryLiteral;
  if (firstArg && !Array.isArray(firstArg) && firstArg.type === 'function' && firstArg.name === '=') {
    // It's an assignment, update the right side of the expression
    firstArg.args[1] = queryLiteral;
  } else {
    // It's a simple query, replace the first argument
    cmd.args[0] = queryLiteral;
  }
};

/**
 * Sets, updates, or removes the target field for the RERANK command.
 * This refers to the `targetField =` portion of the command, which specifies                                  │
 * the new column where the rerank score will be stored.                                                       │
 *
 * @param cmd The RERANK command AST node to modify.
 * @param target The name of the target field to set. If `null` the assignment is removed.
 */
exports.setQuery = setQuery;
const setTargetField = (cmd, target) => {
  const firstArg = cmd.args[0];
  const isAssignment = firstArg && !Array.isArray(firstArg) && firstArg.type === 'function' && firstArg.name === '=';

  // Case 1: Set a new target field
  if (target !== null) {
    const newTargetColumn = _builder.Builder.expression.column(target);
    if (isAssignment) {
      // An assignment already exists => update the target field
      firstArg.args[0] = newTargetColumn;
    } else {
      // No assignment exists => create one
      const queryLiteral = cmd.query;
      const assignment = _builder.Builder.expression.func.binary('=', [newTargetColumn, queryLiteral]);
      cmd.args[0] = assignment;
    }
    cmd.targetField = newTargetColumn;
  }
  // Case 2: Remove the target field
  else {
    if (isAssignment) {
      // An assignment exists, keep only the query
      const queryLiteral = cmd.query;
      cmd.args[0] = queryLiteral;
    }
    // If no assignment exists, do nothing
    cmd.targetField = undefined;
  }
};

/**
 * Sets or updates the fields to be used for reranking in the ON clause.
 *
 * @param cmd The RERANK command AST node to modify.
 * @param fields An array of field names or field nodes.
 */
exports.setTargetField = setTargetField;
const setFields = (cmd, fields) => {
  if (typeof fields[0] === 'string') {
    fields = fields.map(fieldOrFieldName => {
      return typeof fieldOrFieldName === 'string' ? _builder.Builder.expression.column(fieldOrFieldName) : fieldOrFieldName;
    });
  }
  const isOnOption = arg => !!arg && !Array.isArray(arg) && arg.type === 'option' && arg.name === 'on';
  const onOption = cmd.args.find(isOnOption);
  if (!onOption) {
    throw new Error('RERANK command must have a ON option');
  }
  cmd.fields.length = 0;
  cmd.fields.push(...fields);
  onOption.args.length = 0;
  onOption.args.push(...fields);
};

/**
 * Sets a parameter in the WITH clause of the RERANK command (e.g., 'inference_id').
 * If the parameter already exists, its value is updated. Otherwise, it is added.
 *
 * @param cmd The RERANK command AST node to modify.
 * @param key The name of the parameter to set.
 * @param value The value of the parameter.
 */
exports.setFields = setFields;
const setWithParameter = (cmd, key, value) => {
  // Converts a value to an appropriate ESQLExpression based on its type
  const toExpression = val => {
    if (typeof val === 'string') {
      return val.startsWith('?') ? _builder.Builder.param.build(val) : _builder.Builder.expression.literal.string(val);
    }
    return val;
  };
  const isWithOption = arg => !!arg && !Array.isArray(arg) && arg.type === 'option' && arg.name === 'with';

  // Validates and retrieves the map from a WITH option
  const getWithOptionMap = withOption => {
    const mapArg = withOption.args[0];
    if (!mapArg || typeof mapArg === 'string' || Array.isArray(mapArg) || mapArg.type !== 'map') {
      throw new Error('WITH option must contain a map');
    }
    return mapArg;
  };

  // Normalizes a key for comparison by removing quotes
  const normalizeKey = keyValue => {
    var _keyValue$replace;
    return (_keyValue$replace = keyValue === null || keyValue === void 0 ? void 0 : keyValue.replace(/"/g, '')) !== null && _keyValue$replace !== void 0 ? _keyValue$replace : '';
  };

  // Checks if an entry in the map has the specified key
  // normalizeKey avoid cases like: "inference_id" === "'inferenceId"' -> this is false
  const getExistingEntry = entry => {
    var _entry$key$valueUnquo;
    return normalizeKey((_entry$key$valueUnquo = entry.key.valueUnquoted) !== null && _entry$key$valueUnquo !== void 0 ? _entry$key$valueUnquo : entry.key.value) === key;
  };

  // Creates a new map entry with the given key and value
  const createMapEntry = (entryKey, entryValue) => ({
    type: 'map-entry',
    name: 'map-entry',
    key: _builder.Builder.expression.literal.string(entryKey),
    value: entryValue,
    location: {
      min: 0,
      max: 0
    },
    text: '',
    incomplete: false
  });
  const withOption = cmd.args.find(isWithOption);
  if (!withOption) {
    throw new Error('RERANK command must have a WITH option');
  }
  const valueExpression = toExpression(value);
  const map = getWithOptionMap(withOption);
  const existingEntry = map.entries.find(getExistingEntry);
  if (existingEntry) {
    existingEntry.value = valueExpression;
  } else {
    map.entries.push(createMapEntry(key, valueExpression));
  }
};
exports.setWithParameter = setWithParameter;