"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateSavedSearch = updateSavedSearch;
var _lodash = require("lodash");
var _data_sources = require("../../../../../common/data_sources");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Updates the saved search with a given data view & Appstate
 * Is executed on every change of those, for making sure the saved search is
 * up to date before fetching data and persisting or sharing
 * @param savedSearch
 * @param dataView
 * @param state
 * @param globalStateContainer
 * @param services
 * @param useFilterAndQueryServices - when true data services are being used for updating filter + query
 */
function updateSavedSearch({
  savedSearch,
  dataView,
  initialInternalState,
  appState,
  globalState,
  services,
  useFilterAndQueryServices = false
}) {
  var _savedSearch$searchSo;
  if (dataView && ((_savedSearch$searchSo = savedSearch.searchSource.getField('index')) === null || _savedSearch$searchSo === void 0 ? void 0 : _savedSearch$searchSo.id) !== dataView.id) {
    savedSearch.searchSource.setField('index', dataView);
    if (!dataView.isPersisted()) {
      savedSearch.usesAdHocDataView = true;
    }
  }
  if (initialInternalState !== null && initialInternalState !== void 0 && initialInternalState.visContext) {
    savedSearch.visContext = initialInternalState.visContext;
  }
  if (initialInternalState !== null && initialInternalState !== void 0 && initialInternalState.controlGroupJson) {
    savedSearch.controlGroupJson = initialInternalState.controlGroupJson;
  }
  if (useFilterAndQueryServices) {
    savedSearch.searchSource.setField('query', services.data.query.queryString.getQuery()).setField('filter', services.data.query.filterManager.getFilters());
  } else if (appState) {
    var _appState$query;
    const appFilters = appState.filters ? (0, _lodash.cloneDeep)(appState.filters) : [];
    const globalFilters = globalState !== null && globalState !== void 0 && globalState.filters ? (0, _lodash.cloneDeep)(globalState.filters) : [];
    savedSearch.searchSource.setField('query', (_appState$query = appState.query) !== null && _appState$query !== void 0 ? _appState$query : undefined).setField('filter', [...globalFilters, ...appFilters]);
  }
  if (appState) {
    savedSearch.columns = appState.columns || [];
    savedSearch.sort = appState.sort || [];
    if (appState.grid) {
      savedSearch.grid = appState.grid;
    }
    savedSearch.hideChart = appState.hideChart;
    savedSearch.rowHeight = appState.rowHeight;
    savedSearch.headerRowHeight = appState.headerRowHeight;
    savedSearch.rowsPerPage = appState.rowsPerPage;
    savedSearch.sampleSize = appState.sampleSize;
    savedSearch.density = appState.density;
    if (appState.viewMode) {
      savedSearch.viewMode = appState.viewMode;
    }
    if (typeof appState.breakdownField !== 'undefined') {
      savedSearch.breakdownField = appState.breakdownField;
    } else if (savedSearch.breakdownField) {
      savedSearch.breakdownField = '';
    }
    savedSearch.hideAggregatedPreview = appState.hideAggregatedPreview;

    // add a flag here to identify ES|QL queries
    // these should be filtered out from the visualize editor
    const isEsqlMode = (0, _data_sources.isDataSourceType)(appState.dataSource, _data_sources.DataSourceType.Esql);
    if (savedSearch.isTextBasedQuery || isEsqlMode) {
      savedSearch.isTextBasedQuery = isEsqlMode;
    }
  }
  const timeRange = globalState === null || globalState === void 0 ? void 0 : globalState.timeRange;
  const refreshInterval = globalState === null || globalState === void 0 ? void 0 : globalState.refreshInterval;
  savedSearch.timeRange = timeRange && (savedSearch.timeRestore || savedSearch.timeRange) ? {
    from: timeRange.from,
    to: timeRange.to
  } : undefined;
  savedSearch.refreshInterval = refreshInterval && (savedSearch.timeRestore || savedSearch.refreshInterval) ? {
    value: refreshInterval.value,
    pause: refreshInterval.pause
  } : undefined;
  return savedSearch;
}