"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateRuntimeField = exports.registerUpdateRuntimeFieldRouteLegacy = exports.registerUpdateRuntimeFieldRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _error = require("../../../error");
var _handle_errors = require("../util/handle_errors");
var _schemas = require("../../../schemas");
var _constants = require("../../../constants");
var _response_formatter = require("./response_formatter");
var _schema = require("../../schema");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const updateRuntimeField = async ({
  dataViewsService,
  usageCollection,
  counterName,
  id,
  name,
  runtimeField
}) => {
  usageCollection === null || usageCollection === void 0 ? void 0 : usageCollection.incrementCounter({
    counterName
  });
  const dataView = await dataViewsService.getDataViewLazy(id);
  const existingRuntimeField = dataView.getRuntimeField(name);
  if (!existingRuntimeField) {
    throw new _error.ErrorIndexPatternFieldNotFound(id, name);
  }
  dataView.removeRuntimeField(name);
  const fields = await dataView.addRuntimeField(name, {
    ...existingRuntimeField,
    ...runtimeField
  });
  await dataViewsService.updateSavedObject(dataView);
  return {
    dataView,
    fields
  };
};
exports.updateRuntimeField = updateRuntimeField;
const updateRuntimeFieldRouteFactory = (path, serviceKey, description) => (router, getStartServices, usageCollection) => {
  router.versioned.post({
    path,
    access: 'public',
    description,
    security: {
      authz: {
        requiredPrivileges: ['indexPatterns:manage']
      }
    }
  }).addVersion({
    version: _constants.INITIAL_REST_VERSION,
    validate: {
      request: {
        params: _configSchema.schema.object({
          id: _configSchema.schema.string({
            minLength: 1,
            maxLength: 1_000
          }),
          name: _configSchema.schema.string({
            minLength: 1,
            maxLength: 1_000
          })
        }),
        body: _configSchema.schema.object({
          name: _configSchema.schema.never(),
          runtimeField: _schemas.runtimeFieldSchemaUpdate
        })
      },
      response: {
        200: {
          body: _schema.runtimeResponseSchema
        }
      }
    }
  }, (0, _handle_errors.handleErrors)(async (ctx, req, res) => {
    const core = await ctx.core;
    const savedObjectsClient = core.savedObjects.client;
    const elasticsearchClient = core.elasticsearch.client.asCurrentUser;
    const [,, {
      dataViewsServiceFactory
    }] = await getStartServices();
    const dataViewsService = await dataViewsServiceFactory(savedObjectsClient, elasticsearchClient, req);
    const id = req.params.id;
    const name = req.params.name;
    const runtimeField = req.body.runtimeField;
    const {
      dataView,
      fields
    } = await updateRuntimeField({
      dataViewsService,
      usageCollection,
      counterName: `${req.route.method} ${path}`,
      id,
      name,
      runtimeField
    });
    const response = await (0, _response_formatter.responseFormatter)({
      serviceKey,
      dataView,
      fields
    });
    return res.ok(response);
  }));
};
const registerUpdateRuntimeFieldRoute = exports.registerUpdateRuntimeFieldRoute = updateRuntimeFieldRouteFactory(_constants.SPECIFIC_RUNTIME_FIELD_PATH, _constants.SERVICE_KEY, _constants.UPDATE_RUNTIME_FIELD_DESCRIPTION);
const registerUpdateRuntimeFieldRouteLegacy = exports.registerUpdateRuntimeFieldRouteLegacy = updateRuntimeFieldRouteFactory(_constants.SPECIFIC_RUNTIME_FIELD_PATH_LEGACY, _constants.SERVICE_KEY_LEGACY);