/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

package org.elasticsearch.xpack.esql.expression.function.scalar.convert;

import org.apache.lucene.util.BytesRef;
import org.elasticsearch.common.io.stream.NamedWriteableRegistry;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.compute.ann.ConvertEvaluator;
import org.elasticsearch.compute.ann.Fixed;
import org.elasticsearch.xpack.esql.core.expression.Expression;
import org.elasticsearch.xpack.esql.core.tree.NodeInfo;
import org.elasticsearch.xpack.esql.core.tree.Source;
import org.elasticsearch.xpack.esql.core.type.DataType;
import org.elasticsearch.xpack.esql.expression.function.Example;
import org.elasticsearch.xpack.esql.expression.function.FunctionInfo;
import org.elasticsearch.xpack.esql.expression.function.Param;

import java.io.IOException;
import java.util.List;
import java.util.Map;

import static org.elasticsearch.xpack.esql.core.type.DataType.GEOHASH;
import static org.elasticsearch.xpack.esql.core.type.DataType.GEOHEX;
import static org.elasticsearch.xpack.esql.core.type.DataType.GEOTILE;
import static org.elasticsearch.xpack.esql.core.type.DataType.GEO_POINT;
import static org.elasticsearch.xpack.esql.core.type.DataType.GEO_SHAPE;
import static org.elasticsearch.xpack.esql.core.type.DataType.KEYWORD;
import static org.elasticsearch.xpack.esql.core.type.DataType.TEXT;
import static org.elasticsearch.xpack.esql.type.EsqlDataTypeConverter.geoGridToShape;
import static org.elasticsearch.xpack.esql.type.EsqlDataTypeConverter.stringToGeo;

public class ToGeoShape extends AbstractConvertFunction {
    public static final NamedWriteableRegistry.Entry ENTRY = new NamedWriteableRegistry.Entry(
        Expression.class,
        "ToGeoShape",
        ToGeoShape::new
    );

    private static final Map<DataType, BuildFactory> EVALUATORS = Map.ofEntries(
        Map.entry(GEO_POINT, (source, fieldEval) -> fieldEval),
        Map.entry(GEO_SHAPE, (source, fieldEval) -> fieldEval),
        Map.entry(GEOHASH, (source, fieldEval) -> new ToGeoShapeFromGeoGridEvaluator.Factory(source, fieldEval, GEOHASH)),
        Map.entry(GEOTILE, (source, fieldEval) -> new ToGeoShapeFromGeoGridEvaluator.Factory(source, fieldEval, GEOTILE)),
        Map.entry(GEOHEX, (source, fieldEval) -> new ToGeoShapeFromGeoGridEvaluator.Factory(source, fieldEval, GEOHEX)),
        Map.entry(KEYWORD, ToGeoShapeFromStringEvaluator.Factory::new),
        Map.entry(TEXT, ToGeoShapeFromStringEvaluator.Factory::new)
    );

    @FunctionInfo(
        returnType = "geo_shape",
        description = """
            Converts an input value to a `geo_shape` value.
            A string will only be successfully converted if it respects the
            {wikipedia}/Well-known_text_representation_of_geometry[WKT] format.""",
        examples = @Example(file = "spatial_shapes", tag = "to_geoshape-str")
    )
    public ToGeoShape(
        Source source,
        @Param(
            name = "field",
            type = { "geo_point", "geo_shape", "geohash", "geohex", "geotile", "keyword", "text" },
            description = "Input value. The input can be a single- or multi-valued column or an expression."
        ) Expression field
    ) {
        super(source, field);
    }

    private ToGeoShape(StreamInput in) throws IOException {
        super(in);
    }

    @Override
    public String getWriteableName() {
        return ENTRY.name;
    }

    @Override
    protected Map<DataType, BuildFactory> factories() {
        return EVALUATORS;
    }

    @Override
    public DataType dataType() {
        return GEO_SHAPE;
    }

    @Override
    public Expression replaceChildren(List<Expression> newChildren) {
        return new ToGeoShape(source(), newChildren.get(0));
    }

    @Override
    protected NodeInfo<? extends Expression> info() {
        return NodeInfo.create(this, ToGeoShape::new, field());
    }

    @ConvertEvaluator(extraName = "FromString", warnExceptions = { IllegalArgumentException.class })
    static BytesRef fromKeyword(BytesRef in) {
        return stringToGeo(in.utf8ToString());
    }

    @ConvertEvaluator(extraName = "FromGeoGrid", warnExceptions = { IllegalArgumentException.class })
    static BytesRef fromGeoGrid(long in, @Fixed DataType dataType) {
        return geoGridToShape(in, dataType);
    }
}
