/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.file;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.AccessDeniedException;
import java.nio.file.ClosedWatchServiceException;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileTime;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.common.Randomness;
import org.elasticsearch.common.component.AbstractLifecycleComponent;

public abstract class AbstractFileWatchingService
extends AbstractLifecycleComponent {
    private static final Logger logger = LogManager.getLogger(AbstractFileWatchingService.class);
    private static final int REGISTER_RETRY_COUNT = 5;
    private static final int ACCESS_DENIED_RETRY_COUNT = 5;
    private final Path settingsDir;
    private final Map<Path, FileUpdateState> fileUpdateState = new HashMap<Path, FileUpdateState>();
    private WatchService watchService;
    private Thread watcherThread;
    private WatchKey settingsDirWatchKey;
    private WatchKey configDirWatchKey;

    public AbstractFileWatchingService(Path settingsDir) {
        if (this.filesExists(settingsDir) && !this.filesIsDirectory(settingsDir)) {
            throw new IllegalArgumentException("settingsDir should be a directory");
        }
        this.settingsDir = settingsDir;
    }

    protected abstract void processFileChanges(Path var1) throws InterruptedException, ExecutionException, IOException;

    protected abstract void processInitialFilesMissing() throws InterruptedException, ExecutionException, IOException;

    protected void processFileOnServiceStart(Path file) throws IOException, ExecutionException, InterruptedException {
        this.processFileChanges(file);
    }

    public final Path watchedFileDir() {
        return this.settingsDir;
    }

    @Override
    protected void doStart() {
        this.startWatcher();
    }

    @Override
    protected void doStop() {
        logger.debug("Stopping file watching service");
        this.stopWatcher();
    }

    @Override
    protected final void doClose() {
    }

    public final boolean watching() {
        return this.watcherThread != null;
    }

    FileUpdateState readFileUpdateState(Path path) throws IOException, InterruptedException {
        int retryCount = 0;
        while (true) {
            try {
                BasicFileAttributes attr = this.filesReadAttributes(path, BasicFileAttributes.class);
                return new FileUpdateState(attr.lastModifiedTime().toMillis(), path.toRealPath(new LinkOption[0]).toString(), attr.fileKey());
            }
            catch (NoSuchFileException e) {
                return null;
            }
            catch (AccessDeniedException e) {
                if (retryCount == 4) {
                    throw e;
                }
                logger.debug("Could not read file state [{}] attempt [{}]", (Object)path, (Object)retryCount);
                Thread.sleep(this.retryDelayMillis(retryCount));
                ++retryCount;
                continue;
            }
            break;
        }
    }

    final boolean fileChanged(Path path) throws IOException, InterruptedException {
        FileUpdateState newFileState = this.readFileUpdateState(path);
        if (newFileState == null) {
            this.fileUpdateState.remove(path);
            return false;
        }
        FileUpdateState previousUpdateState = this.fileUpdateState.put(path, newFileState);
        return previousUpdateState == null || !previousUpdateState.equals(newFileState);
    }

    protected final synchronized void startWatcher() {
        if (!this.filesExists(this.settingsDir.getParent())) {
            logger.warn("File watcher for [{}] cannot start because parent directory does not exist", (Object)this.settingsDir);
            return;
        }
        logger.info("starting file watcher ...");
        try {
            this.watchService = this.settingsDir.getParent().getFileSystem().newWatchService();
            if (this.filesExists(this.settingsDir)) {
                this.settingsDirWatchKey = this.enableDirectoryWatcher(this.settingsDirWatchKey, this.settingsDir);
            } else {
                logger.debug("watched directory [{}] not found, will watch for its creation...", (Object)this.settingsDir);
            }
            this.configDirWatchKey = this.enableDirectoryWatcher(this.configDirWatchKey, this.settingsDir.getParent());
        }
        catch (Exception e) {
            if (this.watchService != null) {
                try {
                    this.watchService.close();
                }
                catch (Exception ce) {
                    e.addSuppressed(ce);
                }
                finally {
                    this.watchService = null;
                }
            }
            throw new IllegalStateException("unable to launch a new watch service", e);
        }
        this.watcherThread = new Thread(this::watcherThread, "elasticsearch[file-watcher[" + String.valueOf(this.settingsDir) + "]]");
        this.watcherThread.start();
    }

    protected final void watcherThread() {
        try {
            WatchKey key;
            logger.info("file settings service up and running [tid={}]", (Object)Thread.currentThread().getId());
            if (this.filesExists(this.settingsDir)) {
                try (Stream<Path> files = this.filesList(this.settingsDir);){
                    Iterator f = files.iterator();
                    if (!f.hasNext()) {
                        this.processInitialFilesMissing();
                    }
                    do {
                        Path next;
                        FileUpdateState state;
                        if ((state = this.readFileUpdateState(next = (Path)f.next())) == null) continue;
                        this.fileUpdateState.put(next, state);
                        logger.debug("found initial settings file [{}], applying...", (Object)next);
                        this.processOnServiceStart(next);
                    } while (f.hasNext());
                }
            } else {
                this.processInitialFilesMissing();
            }
            block9: while ((key = this.watchService.take()) != null) {
                List<WatchEvent<?>> events = key.pollEvents();
                if (logger.isDebugEnabled()) {
                    logger.debug("Processing events from {}", (Object)key.watchable());
                    events.forEach(e -> logger.debug("{}:{}", e.kind(), e.context()));
                }
                key.reset();
                if (key == this.settingsDirWatchKey) {
                    Set changedPaths = events.stream().map(event -> this.settingsDir.resolve(event.context().toString())).collect(Collectors.toSet());
                    for (Path changedPath : changedPaths) {
                        if (this.filesIsDirectory(changedPath) && this.filesIsSymbolicLink(changedPath)) {
                            this.reprocessAllChangedFilesInSettingsDir();
                            continue block9;
                        }
                        if (!this.fileChanged(changedPath)) continue;
                        this.process(changedPath);
                    }
                    continue;
                }
                if (key == this.configDirWatchKey) {
                    if (this.filesExists(this.settingsDir)) {
                        logger.debug("Re-registering settings directory watch");
                        this.settingsDirWatchKey = this.enableDirectoryWatcher(this.settingsDirWatchKey, this.settingsDir);
                        this.reprocessAllChangedFilesInSettingsDir();
                        continue;
                    }
                    if (this.settingsDirWatchKey == null) continue;
                    this.settingsDirWatchKey.cancel();
                    continue;
                }
                logger.warn("Received events for unknown watch key {}", (Object)key);
            }
        }
        catch (InterruptedException | ClosedWatchServiceException expected) {
            logger.info("shutting down watcher thread");
        }
        catch (Exception e2) {
            logger.error("shutting down watcher thread with exception", (Throwable)e2);
        }
    }

    private void reprocessAllChangedFilesInSettingsDir() throws IOException, InterruptedException {
        try (Stream<Path> files = this.filesList(this.settingsDir);){
            Iterator f = files.iterator();
            while (f.hasNext()) {
                Path file = (Path)f.next();
                if (!this.fileChanged(file)) continue;
                this.process(file);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    protected final synchronized void stopWatcher() {
        if (this.watching()) {
            logger.debug("stopping watcher ...");
            try (WatchService ws = this.watchService;){
                this.watcherThread.interrupt();
                this.watcherThread.join();
                if (this.configDirWatchKey != null) {
                    this.configDirWatchKey.cancel();
                }
                if (this.settingsDirWatchKey == null) return;
                this.settingsDirWatchKey.cancel();
                return;
            }
            catch (IOException e) {
                logger.warn("encountered exception while closing watch service", (Throwable)e);
                return;
            }
            catch (InterruptedException interruptedException) {
                logger.info("interrupted while closing the watch service", (Throwable)interruptedException);
                return;
            }
            finally {
                this.watcherThread = null;
                this.settingsDirWatchKey = null;
                this.configDirWatchKey = null;
                this.watchService = null;
                logger.info("watcher service stopped");
            }
        } else {
            logger.trace("file watch service already stopped");
        }
    }

    final WatchKey enableDirectoryWatcher(WatchKey previousKey, Path settingsDir) throws IOException, InterruptedException {
        if (previousKey != null) {
            previousKey.cancel();
        }
        int retryCount = 0;
        while (true) {
            try {
                return settingsDir.register(this.watchService, StandardWatchEventKinds.ENTRY_MODIFY, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE);
            }
            catch (IOException e) {
                if (retryCount == 4) {
                    throw e;
                }
                Thread.sleep(this.retryDelayMillis(retryCount));
                ++retryCount;
                continue;
            }
            break;
        }
    }

    void processOnServiceStart(Path file) throws InterruptedException {
        try {
            this.processFileOnServiceStart(file);
        }
        catch (IOException | ExecutionException e) {
            this.onProcessFileChangesException(file, e);
        }
    }

    void process(Path file) throws InterruptedException {
        try {
            this.processFileChanges(file);
        }
        catch (IOException | ExecutionException e) {
            this.onProcessFileChangesException(file, e);
        }
    }

    protected void onProcessFileChangesException(Path file, Exception e) {
        logger.error(() -> "Error processing file notification: " + String.valueOf(file), (Throwable)e);
    }

    long retryDelayMillis(int failedCount) {
        assert (failedCount < 31);
        return 100 * (1 << failedCount) + Randomness.get().nextInt(10);
    }

    protected abstract boolean filesExists(Path var1);

    protected abstract boolean filesIsDirectory(Path var1);

    protected abstract boolean filesIsSymbolicLink(Path var1);

    protected abstract <A extends BasicFileAttributes> A filesReadAttributes(Path var1, Class<A> var2) throws IOException;

    protected abstract Stream<Path> filesList(Path var1) throws IOException;

    protected abstract Path filesSetLastModifiedTime(Path var1, FileTime var2) throws IOException;

    protected abstract InputStream filesNewInputStream(Path var1) throws IOException;

    private record FileUpdateState(long timestamp, String path, Object fileKey) {
    }
}

