"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.checkAccess = checkAccess;
exports.checkAccessBulk = checkAccessBulk;
exports.getDataStream = getDataStream;
exports.getDataStreamLifecycle = getDataStreamLifecycle;
exports.getDataStreamSettings = getDataStreamSettings;
exports.getDefaultRetentionValue = getDefaultRetentionValue;
exports.getFailureStore = getFailureStore;
exports.getFailureStoreCreationDate = getFailureStoreCreationDate;
exports.getFailureStoreMeteringSize = getFailureStoreMeteringSize;
exports.getFailureStoreSize = getFailureStoreSize;
exports.getFailureStoreStats = getFailureStoreStats;
exports.getUnmanagedElasticsearchAssetDetails = getUnmanagedElasticsearchAssetDetails;
exports.getUnmanagedElasticsearchAssets = getUnmanagedElasticsearchAssets;
exports.updateFailureStore = updateFailureStore;
var _constants = require("../../../common/constants");
var _definition_not_found_error = require("./errors/definition_not_found_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getDataStreamLifecycle(dataStream) {
  if (!dataStream) {
    return {
      error: {
        message: 'Data stream not found'
      }
    };
  }
  if (dataStream.next_generation_managed_by === 'Index Lifecycle Management') {
    return {
      ilm: {
        policy: dataStream.ilm_policy
      }
    };
  }
  if (dataStream.next_generation_managed_by === 'Data stream lifecycle') {
    var _dataStream$lifecycle;
    const retention = (_dataStream$lifecycle = dataStream.lifecycle) === null || _dataStream$lifecycle === void 0 ? void 0 : _dataStream$lifecycle.data_retention;
    return {
      dsl: {
        data_retention: retention ? String(retention) : undefined
      }
    };
  }
  if (dataStream.next_generation_managed_by === 'Unmanaged') {
    return {
      disabled: {}
    };
  }
  return {
    error: {
      message: `Unknown data stream lifecycle state [${dataStream.next_generation_managed_by}]`
    }
  };
}
function getDataStreamSettings(dataStream) {
  var _dataStream$effective, _dataStream$effective2, _dataStream$effective3;
  const settings = {};
  if (dataStream !== null && dataStream !== void 0 && (_dataStream$effective = dataStream.effective_settings.index) !== null && _dataStream$effective !== void 0 && _dataStream$effective.number_of_replicas) {
    settings['index.number_of_replicas'] = {
      value: Number(dataStream.effective_settings.index.number_of_replicas)
    };
  }
  if (dataStream !== null && dataStream !== void 0 && (_dataStream$effective2 = dataStream.effective_settings.index) !== null && _dataStream$effective2 !== void 0 && _dataStream$effective2.number_of_shards) {
    settings['index.number_of_shards'] = {
      value: Number(dataStream.effective_settings.index.number_of_shards)
    };
  }
  if (dataStream !== null && dataStream !== void 0 && (_dataStream$effective3 = dataStream.effective_settings.index) !== null && _dataStream$effective3 !== void 0 && _dataStream$effective3.refresh_interval) {
    settings['index.refresh_interval'] = {
      value: dataStream.effective_settings.index.refresh_interval
    };
  }
  return settings;
}
async function getUnmanagedElasticsearchAssets({
  dataStream,
  scopedClusterClient
}) {
  var _dataStream$indices$a, _currentIndex$writeIn, _currentIndex$writeIn2;
  // retrieve linked index template, component template and ingest pipeline
  const templateName = dataStream.template;
  const componentTemplates = [];
  const template = await scopedClusterClient.asCurrentUser.indices.getIndexTemplate({
    name: templateName
  });
  if (template.index_templates.length) {
    var _template$index_templ;
    (_template$index_templ = template.index_templates[0].index_template.composed_of) === null || _template$index_templ === void 0 ? void 0 : _template$index_templ.forEach(componentTemplateName => {
      componentTemplates.push(componentTemplateName);
    });
  }
  const writeIndexName = (_dataStream$indices$a = dataStream.indices.at(-1)) === null || _dataStream$indices$a === void 0 ? void 0 : _dataStream$indices$a.index_name;
  const currentIndex = await scopedClusterClient.asCurrentUser.indices.get({
    index: writeIndexName
  });
  const ingestPipelineId = (_currentIndex$writeIn = currentIndex[writeIndexName].settings) === null || _currentIndex$writeIn === void 0 ? void 0 : (_currentIndex$writeIn2 = _currentIndex$writeIn.index) === null || _currentIndex$writeIn2 === void 0 ? void 0 : _currentIndex$writeIn2.default_pipeline;
  return {
    ingestPipeline: ingestPipelineId,
    componentTemplates,
    indexTemplate: templateName,
    dataStream: dataStream.name
  };
}
async function fetchComponentTemplate(scopedClusterClient, name) {
  try {
    var _response$component_t;
    const response = await scopedClusterClient.asCurrentUser.cluster.getComponentTemplate({
      name
    });
    return (_response$component_t = response.component_templates.find(template => template.name === name)) !== null && _response$component_t !== void 0 ? _response$component_t : {
      name,
      component_template: undefined
    };
  } catch (e) {
    var _e$meta;
    if (((_e$meta = e.meta) === null || _e$meta === void 0 ? void 0 : _e$meta.statusCode) === 404) {
      return {
        name,
        component_template: undefined
      };
    }
    throw e;
  }
}
async function fetchComponentTemplates(scopedClusterClient, names, allIndexTemplates) {
  const templates = await Promise.all(names.map(name => fetchComponentTemplate(scopedClusterClient, name)));
  return templates.filter(template => template !== undefined).map(componentTemplate => ({
    ...componentTemplate,
    used_by: allIndexTemplates.filter(template => {
      var _template$index_templ2;
      return (_template$index_templ2 = template.index_template.composed_of) === null || _template$index_templ2 === void 0 ? void 0 : _template$index_templ2.includes(componentTemplate.name);
    }).map(template => template.name)
  }));
}
async function fetchIngestPipeline(scopedClusterClient, pipelineId) {
  if (!pipelineId) return undefined;
  const response = await scopedClusterClient.asCurrentUser.ingest.getPipeline({
    id: pipelineId
  });
  return {
    ...response[pipelineId],
    name: pipelineId
  };
}
async function getUnmanagedElasticsearchAssetDetails({
  scopedClusterClient,
  assets
}) {
  const allIndexTemplates = (await scopedClusterClient.asCurrentUser.indices.getIndexTemplate()).index_templates;
  const [ingestPipeline, componentTemplates, dataStreamResponse] = await Promise.all([fetchIngestPipeline(scopedClusterClient, assets.ingestPipeline), fetchComponentTemplates(scopedClusterClient, assets.componentTemplates, allIndexTemplates), scopedClusterClient.asCurrentUser.indices.getDataStream({
    name: assets.dataStream
  })]);
  const indexTemplate = allIndexTemplates.find(template => template.name === assets.indexTemplate);
  if (!indexTemplate) {
    throw new Error(`Index template ${assets.indexTemplate} not found`);
  }
  return {
    ingestPipeline,
    componentTemplates,
    indexTemplate,
    dataStream: dataStreamResponse.data_streams[0]
  };
}
async function checkAccess({
  name,
  scopedClusterClient
}) {
  return checkAccessBulk({
    names: [name],
    scopedClusterClient
  }).then(privileges => privileges[name]);
}
async function checkAccessBulk({
  names,
  scopedClusterClient
}) {
  if (!names.length) {
    return {};
  }
  const hasPrivilegesResponse = await scopedClusterClient.asCurrentUser.security.hasPrivileges({
    index: [{
      names,
      privileges: ['read', 'write']
    }]
  });
  return Object.fromEntries(names.map(name => {
    const hasReadAccess = hasPrivilegesResponse.index[name].read === true;
    const hasWriteAccess = hasPrivilegesResponse.index[name].write === true;
    return [name, {
      read: hasReadAccess,
      write: hasWriteAccess
    }];
  }));
}
async function getDataStream({
  name,
  scopedClusterClient
}) {
  let dataStream;
  try {
    const response = await scopedClusterClient.asCurrentUser.indices.getDataStream({
      name
    });
    dataStream = response.data_streams[0];
  } catch (e) {
    var _e$meta2;
    if (((_e$meta2 = e.meta) === null || _e$meta2 === void 0 ? void 0 : _e$meta2.statusCode) === 404) {
      // fall through and throw not found
    } else {
      throw e;
    }
  }
  if (!dataStream) {
    throw new _definition_not_found_error.DefinitionNotFoundError(`Stream definition for ${name} not found.`);
  }
  return dataStream;
}
async function getDefaultRetentionValue({
  scopedClusterClient
}) {
  let defaultRetention;
  try {
    var _persistent$data_stre, _persistent$data_stre2, _persistent$data_stre3, _defaults$data_stream, _defaults$data_stream2, _defaults$data_stream3;
    const {
      persistent,
      defaults
    } = await scopedClusterClient.asCurrentUser.cluster.getSettings({
      include_defaults: true
    });
    const persistentDSRetention = persistent === null || persistent === void 0 ? void 0 : (_persistent$data_stre = persistent.data_streams) === null || _persistent$data_stre === void 0 ? void 0 : (_persistent$data_stre2 = _persistent$data_stre.lifecycle) === null || _persistent$data_stre2 === void 0 ? void 0 : (_persistent$data_stre3 = _persistent$data_stre2.retention) === null || _persistent$data_stre3 === void 0 ? void 0 : _persistent$data_stre3.failures_default;
    const defaultsDSRetention = defaults === null || defaults === void 0 ? void 0 : (_defaults$data_stream = defaults.data_streams) === null || _defaults$data_stream === void 0 ? void 0 : (_defaults$data_stream2 = _defaults$data_stream.lifecycle) === null || _defaults$data_stream2 === void 0 ? void 0 : (_defaults$data_stream3 = _defaults$data_stream2.retention) === null || _defaults$data_stream3 === void 0 ? void 0 : _defaults$data_stream3.failures_default;
    defaultRetention = persistentDSRetention !== null && persistentDSRetention !== void 0 ? persistentDSRetention : defaultsDSRetention;
  } catch (e) {
    var _e$meta3;
    if (((_e$meta3 = e.meta) === null || _e$meta3 === void 0 ? void 0 : _e$meta3.statusCode) === 403) {
      // if user doesn't have permissions to read cluster settings, we just return undefined
    } else {
      throw e;
    }
  }
  return defaultRetention;
}
async function getFailureStore({
  name,
  scopedClusterClient,
  isServerless
}) {
  var _dataStream$failure_s, _dataStream$failure_s2, _dataStream$failure_s3, _dataStream$failure_s4, _dataStream$failure_s5, _dataStream$failure_s6, _dataStream$failure_s7;
  // TODO: remove DataStreamWithFailureStore here and in streams-schema once failure store is added to the IndicesDataStream type
  const dataStream = await getDataStream({
    name,
    scopedClusterClient
  });
  const defaultRetentionPeriod = ((_dataStream$failure_s = dataStream.failure_store) === null || _dataStream$failure_s === void 0 ? void 0 : (_dataStream$failure_s2 = _dataStream$failure_s.lifecycle) === null || _dataStream$failure_s2 === void 0 ? void 0 : _dataStream$failure_s2.retention_determined_by) === 'default_failures_retention' ? (_dataStream$failure_s3 = dataStream.failure_store) === null || _dataStream$failure_s3 === void 0 ? void 0 : (_dataStream$failure_s4 = _dataStream$failure_s3.lifecycle) === null || _dataStream$failure_s4 === void 0 ? void 0 : _dataStream$failure_s4.effective_retention : isServerless ? undefined : await getDefaultRetentionValue({
    scopedClusterClient
  });
  return {
    enabled: !!((_dataStream$failure_s5 = dataStream.failure_store) !== null && _dataStream$failure_s5 !== void 0 && _dataStream$failure_s5.enabled),
    retentionPeriod: {
      custom: (_dataStream$failure_s6 = dataStream.failure_store) === null || _dataStream$failure_s6 === void 0 ? void 0 : (_dataStream$failure_s7 = _dataStream$failure_s6.lifecycle) === null || _dataStream$failure_s7 === void 0 ? void 0 : _dataStream$failure_s7.data_retention,
      default: defaultRetentionPeriod
    }
  };
}
async function getFailureStoreStats({
  name,
  scopedClusterClient,
  isServerless
}) {
  const failureStoreDocs = isServerless ? await getFailureStoreMeteringSize({
    name,
    scopedClusterClient
  }) : await getFailureStoreSize({
    name,
    scopedClusterClient
  });
  const creationDate = await getFailureStoreCreationDate({
    name,
    scopedClusterClient
  });
  return {
    size: failureStoreDocs === null || failureStoreDocs === void 0 ? void 0 : failureStoreDocs.total_size_in_bytes,
    count: failureStoreDocs === null || failureStoreDocs === void 0 ? void 0 : failureStoreDocs.count,
    creationDate
  };
}
async function getFailureStoreSize({
  name,
  scopedClusterClient
}) {
  try {
    var _response$_all, _response$_all$total;
    const response = await scopedClusterClient.asCurrentUser.indices.stats({
      index: `${name}${_constants.FAILURE_STORE_SELECTOR}`,
      metric: ['docs'],
      forbid_closed_indices: false
    });
    const docsStats = response === null || response === void 0 ? void 0 : (_response$_all = response._all) === null || _response$_all === void 0 ? void 0 : (_response$_all$total = _response$_all.total) === null || _response$_all$total === void 0 ? void 0 : _response$_all$total.docs;
    return {
      count: (docsStats === null || docsStats === void 0 ? void 0 : docsStats.count) || 0,
      total_size_in_bytes: (docsStats === null || docsStats === void 0 ? void 0 : docsStats.total_size_in_bytes) || 0
    };
  } catch (e) {
    var _e$meta4;
    if (((_e$meta4 = e.meta) === null || _e$meta4 === void 0 ? void 0 : _e$meta4.statusCode) === 404) {
      return undefined;
    } else {
      throw e;
    }
  }
}
async function getFailureStoreMeteringSize({
  name,
  scopedClusterClient
}) {
  try {
    var _response$_total, _response$_total2;
    const response = await scopedClusterClient.asSecondaryAuthUser.transport.request({
      method: 'GET',
      path: `/_metering/stats/${name}${_constants.FAILURE_STORE_SELECTOR}`
    });
    return {
      count: ((_response$_total = response._total) === null || _response$_total === void 0 ? void 0 : _response$_total.num_docs) || 0,
      total_size_in_bytes: ((_response$_total2 = response._total) === null || _response$_total2 === void 0 ? void 0 : _response$_total2.size_in_bytes) || 0
    };
  } catch (e) {
    var _e$meta5;
    if (((_e$meta5 = e.meta) === null || _e$meta5 === void 0 ? void 0 : _e$meta5.statusCode) === 404) {
      return undefined;
    } else {
      throw e;
    }
  }
}
async function getFailureStoreCreationDate({
  name,
  scopedClusterClient
}) {
  let age;
  try {
    const response = await scopedClusterClient.asCurrentUser.indices.explainDataLifecycle({
      index: `${name}${_constants.FAILURE_STORE_SELECTOR}`
    });
    const indices = response.indices;
    if (indices && typeof indices === 'object') {
      const firstIndex = Object.values(indices)[0];
      age = firstIndex === null || firstIndex === void 0 ? void 0 : firstIndex.index_creation_date_millis;
    }
    return age || undefined;
  } catch (e) {
    var _e$meta6;
    if (((_e$meta6 = e.meta) === null || _e$meta6 === void 0 ? void 0 : _e$meta6.statusCode) === 404) {
      return undefined;
    } else {
      throw e;
    }
  }
}
async function updateFailureStore({
  name,
  enabled,
  customRetentionPeriod,
  scopedClusterClient,
  isServerless
}) {
  try {
    await scopedClusterClient.asCurrentUser.indices.putDataStreamOptions({
      name,
      failure_store: {
        enabled,
        lifecycle: {
          data_retention: customRetentionPeriod,
          ...(isServerless ? {} : {
            enabled
          })
        }
      }
    }, {
      meta: true
    });
  } catch (error) {
    throw new Error(`Failed to update failure store for stream "${name}": ${error}`);
  }
}