"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.StreamActiveRecord = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * The StreamActiveRecord is responsible for maintaining the change status of a stream
 * And routing change requests (with cascading changes), validation requests and requests to determine Elasticsearch actions
 * to the right hook based on this state
 */
class StreamActiveRecord {
  constructor(definition, dependencies) {
    (0, _defineProperty2.default)(this, "dependencies", void 0);
    (0, _defineProperty2.default)(this, "_definition", void 0);
    (0, _defineProperty2.default)(this, "_changes", {});
    (0, _defineProperty2.default)(this, "_changeStatus", 'unchanged');
    this._definition = definition;
    this.dependencies = dependencies;
  }
  get name() {
    return this._definition.name;
  }
  get definition() {
    return this._definition;
  }
  get changeStatus() {
    return this._changeStatus;
  }

  // Used only when we try to resync the stored State
  markAsUpserted() {
    this._changeStatus = 'upserted';
  }

  // Used only when we failed to create a new stream and need to flip the stream to create deletion actions
  // from the same definition that we attempted to create
  markAsDeleted() {
    this._changeStatus = 'deleted';
  }
  hasChanged() {
    return this.changeStatus !== 'unchanged';
  }
  isDeleted() {
    return this.changeStatus === 'deleted';
  }

  // Apply change asks the stream to consider an incoming change
  // If the change targets the stream it should update itself and fire any cascading changes
  // That might affect its children or its parent
  // In some cases the stream may mark itself as changed in order to perform further validation
  // Or follow up Elasticsearch changes based on a parent or child changing
  async applyChange(change, desiredState, startingState) {
    if (change.type === 'delete') {
      return this.delete(change.name, desiredState, startingState);
    } else {
      return this.upsert(change.definition, desiredState, startingState);
    }
  }
  async delete(target, desiredState, startingState) {
    const {
      cascadingChanges,
      changeStatus
    } = await this.doHandleDeleteChange(target, desiredState, startingState);
    this._changeStatus = changeStatus;
    return cascadingChanges;
  }
  async upsert(definition, desiredState, startingState) {
    const {
      cascadingChanges,
      changeStatus
    } = await this.doHandleUpsertChange(definition, desiredState, startingState);
    this._changeStatus = changeStatus;
    return cascadingChanges;
  }

  // The validate method should have the stream consider, that given this desired state
  // Is the stream in a valid state (also based on the state of its parent and children)
  // And depending on if the stream was deleted or changed
  async validate(desiredState, startingState) {
    try {
      if (this._changeStatus === 'upserted') {
        return await this.doValidateUpsertion(desiredState, startingState);
      } else if (this._changeStatus === 'deleted') {
        return await this.doValidateDeletion(desiredState, startingState);
      }
      return {
        isValid: true,
        errors: []
      };
    } catch (error) {
      return {
        isValid: false,
        errors: [error]
      };
    }
  }

  // If the stream has changed, what actions are needed to change the Elasticsearch
  // resources to match the desired state?
  // If the stream as newly created or deleted we expect all the actions that are needed to
  // create or delete the required resources
  // If the stream was changed, we aim to only update those resources that would be impacted by
  // that particular type of change
  async determineElasticsearchActions(desiredState, startingState, startingStateStream) {
    if (this.changeStatus === 'upserted') {
      if (startingStateStream) {
        return this.doDetermineUpdateActions(desiredState, startingState, startingStateStream);
      } else {
        return this.doDetermineCreateActions(desiredState);
      }
    } else if (this.changeStatus === 'deleted') {
      return this.doDetermineDeleteActions();
    }
    throw new Error('Cannot determine Elasticsearch actions for an unchanged stream');
  }
  toPrintable() {
    return {
      changeStatus: this.changeStatus,
      changes: this.getChanges(),
      definition: this.definition
    };
  }
  clone() {
    // Get a copy of the current definition
    const clonedStream = this.doClone();
    // Carry over any changes not included in the definition
    clonedStream.setChanges(this.getChanges());
    return clonedStream;
  }
  getChanges() {
    return this._changes;
  }
  setChanges(changes) {
    this._changes = {
      ...changes
    };
  }
}
exports.StreamActiveRecord = StreamActiveRecord;