"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateAncestorFields = validateAncestorFields;
exports.validateClassicFields = validateClassicFields;
exports.validateDescendantFields = validateDescendantFields;
exports.validateSystemFields = validateSystemFields;
var _streamsSchema = require("@kbn/streams-schema");
var _malformed_fields_error = require("../errors/malformed_fields_error");
var _logs_layer = require("../component_templates/logs_layer");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function validateAncestorFields({
  ancestors,
  fields
}) {
  for (const ancestor of ancestors) {
    for (const fieldName in fields) {
      if (!Object.hasOwn(fields, fieldName)) {
        continue;
      }
      if (Object.entries(ancestor.ingest.wired.fields).some(([ancestorFieldName, attr]) => attr.type !== fields[fieldName].type && ancestorFieldName === fieldName)) {
        throw new _malformed_fields_error.MalformedFieldsError(`Field ${fieldName} is already defined with incompatible type in the parent stream ${ancestor.name}`);
      }
      if (!_streamsSchema.namespacePrefixes.some(prefix => fieldName.startsWith(prefix)) && !_streamsSchema.keepFields.includes(fieldName)) {
        throw new _malformed_fields_error.MalformedFieldsError(`Field ${fieldName} is not allowed to be defined as it doesn't match the namespaced ECS or OTel schema.`);
      }
      for (const prefix of _streamsSchema.namespacePrefixes) {
        const prefixedName = `${prefix}${fieldName}`;
        if (Object.hasOwn(fields, prefixedName) || Object.hasOwn(ancestor.ingest.wired.fields, prefixedName)) {
          throw new _malformed_fields_error.MalformedFieldsError(`Field ${fieldName} is an automatic alias of ${prefixedName} because of otel compat mode`);
        }
      }
      // check the otelMappings - they are aliases and are not allowed to have the same name as a field
      if (fieldName in _logs_layer.baseMappings) {
        throw new _malformed_fields_error.MalformedFieldsError(`Field ${fieldName} is an automatic alias of another field because of otel compat mode`);
      }
    }
  }
}
function validateSystemFields(definition) {
  if ((0, _streamsSchema.isRoot)(definition.name)) {
    // the root stream is allowed to have system fields
    return;
  }
  // child streams are not supposed to have system fields
  if (Object.values(definition.ingest.wired.fields).some(field => field.type === 'system')) {
    throw new _malformed_fields_error.MalformedFieldsError(`Stream ${definition.name} is not allowed to have system fields`);
  }
}
function validateClassicFields(definition) {
  if (Object.values(definition.ingest.classic.field_overrides || {}).some(field => field.type === 'system')) {
    throw new _malformed_fields_error.MalformedFieldsError(`Stream ${definition.name} is not allowed to have system fields`);
  }
}
function validateDescendantFields({
  descendants,
  fields
}) {
  for (const descendant of descendants) {
    for (const fieldName in fields) {
      if (Object.hasOwn(fields, fieldName) && Object.entries(descendant.ingest.wired.fields).some(([descendantFieldName, attr]) => attr.type !== fields[fieldName].type && descendantFieldName === fieldName)) {
        throw new _malformed_fields_error.MalformedFieldsError(`Field ${fieldName} is already defined with incompatible type in the child stream ${descendant.name}`);
      }
    }
  }
}