"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FeatureClient = void 0;
var _esQuery = require("@kbn/es-query");
var _objectHash = _interopRequireDefault(require("object-hash"));
var _feature_not_found_error = require("../errors/feature_not_found_error");
var _fields = require("./fields");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getFeatureUuid(streamName, featureName) {
  return (0, _objectHash.default)({
    [_fields.STREAM_NAME]: streamName,
    [_fields.FEATURE_NAME]: featureName
  });
}
function fromStorage(link) {
  return {
    name: link[_fields.FEATURE_NAME],
    description: link[_fields.FEATURE_DESCRIPTION],
    filter: link[_fields.FEATURE_FILTER]
  };
}
function toStorage(name, feature) {
  return {
    [_fields.STREAM_NAME]: name,
    [_fields.FEATURE_UUID]: getFeatureUuid(name, feature.name),
    [_fields.FEATURE_NAME]: feature.name,
    [_fields.FEATURE_DESCRIPTION]: feature.description,
    [_fields.FEATURE_FILTER]: feature.filter
  };
}
class FeatureClient {
  constructor(clients) {
    this.clients = clients;
  }
  async syncFeatureList(name, features) {
    const featuresResponse = await this.clients.storageClient.search({
      size: 10_000,
      track_total_hits: false,
      query: {
        bool: {
          filter: [...(0, _esQuery.termQuery)(_fields.STREAM_NAME, name)]
        }
      }
    });
    const existingFeatures = featuresResponse.hits.hits.map(hit => {
      return hit._source;
    });
    const nextFeatures = features.map(feature => {
      return toStorage(name, feature);
    });
    const nextIds = new Set(nextFeatures.map(feature => feature[_fields.FEATURE_UUID]));
    const featuresDeleted = existingFeatures.filter(feature => !nextIds.has(feature[_fields.FEATURE_UUID]));
    const operations = [...featuresDeleted.map(feature => ({
      delete: {
        feature: fromStorage(feature),
        name
      }
    })), ...nextFeatures.map(feature => ({
      index: {
        feature: fromStorage(feature),
        name
      }
    }))];
    if (operations.length) {
      await this.bulk(name, operations);
    }
    return {
      deleted: featuresDeleted.map(feature => fromStorage(feature)),
      indexed: features
    };
  }
  async linkFeature(name, feature) {
    const document = toStorage(name, feature);
    await this.clients.storageClient.index({
      id: document[_fields.FEATURE_UUID],
      document
    });
    return feature;
  }
  async unlinkFeature(name, feature) {
    const id = getFeatureUuid(name, feature.name);
    const {
      result
    } = await this.clients.storageClient.delete({
      id
    });
    if (result === 'not_found') {
      throw new _feature_not_found_error.FeatureNotFoundError(`Feature ${feature.name} not found for stream ${name}`);
    }
  }
  async clean() {
    await this.clients.storageClient.clean();
  }
  async bulk(name, operations) {
    return await this.clients.storageClient.bulk({
      operations: operations.map(operation => {
        if ('index' in operation) {
          const document = toStorage(name, operation.index.feature);
          return {
            index: {
              document,
              _id: document[_fields.FEATURE_UUID]
            }
          };
        }
        const id = getFeatureUuid(name, operation.delete.feature.name);
        return {
          delete: {
            _id: id
          }
        };
      })
    });
  }
  async getFeature(name, featureName) {
    const id = getFeatureUuid(name, featureName);
    const hit = await this.clients.storageClient.get({
      id
    });
    return fromStorage(hit._source);
  }
  async deleteFeature(name, featureName) {
    const id = getFeatureUuid(name, featureName);
    return await this.clients.storageClient.delete({
      id
    });
  }
  async updateFeature(name, feature) {
    const id = getFeatureUuid(name, feature.name);
    return await this.clients.storageClient.index({
      document: toStorage(name, feature),
      id
    });
  }
  async getFeatures(name) {
    const featuresResponse = await this.clients.storageClient.search({
      size: 10_000,
      track_total_hits: true,
      query: {
        bool: {
          filter: [...(0, _esQuery.termQuery)(_fields.STREAM_NAME, name)]
        }
      }
    });
    return {
      hits: featuresResponse.hits.hits.map(hit => fromStorage(hit._source)),
      total: featuresResponse.hits.total.value
    };
  }
}
exports.FeatureClient = FeatureClient;