"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.asTree = asTree;
exports.mergeTrees = mergeTrees;
var _lodash = require("lodash");
var _helpers = require("./helpers");
var _error = require("../error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function asTree({
  root,
  streams,
  include
}) {
  const stream = streams.find(({
    name
  }) => name === root);
  if (!stream) {
    throw new Error(`Could not find stream [${root}]`);
  }
  const routing = (0, _helpers.filterRouting)(stream, include);
  return {
    ...stream,
    request: {
      ...stream.request,
      queries: (0, _helpers.filterQueries)(stream, include),
      stream: {
        ...stream.request.stream,
        ingest: {
          ...stream.request.stream.ingest,
          wired: {
            ...stream.request.stream.ingest.wired,
            fields: (0, _helpers.getFields)(stream, include),
            routing
          }
        }
      }
    },
    children: routing.map(({
      destination
    }) => asTree({
      streams,
      root: destination,
      include: (0, _helpers.includedObjectsFor)(destination, include)
    }))
  };
}

/**
 * merges the root streams provided.
 * this is not called recursively on the children as we currently
 * fail when trying to merge a child that already exists.
 */
function mergeTrees({
  existing,
  incoming
}) {
  assertNoConflicts(existing, incoming);
  const mergedRouting = [...existing.request.stream.ingest.wired.routing, ...incoming.request.stream.ingest.wired.routing];
  const mergedFields = {
    ...existing.request.stream.ingest.wired.fields,
    ...incoming.request.stream.ingest.wired.fields
  };
  const mergedQueries = [...existing.request.queries, ...incoming.request.queries];
  const mergedChildren = [...existing.children, ...incoming.children];
  return {
    type: 'stream',
    name: existing.name,
    request: {
      ...existing.request,
      queries: mergedQueries,
      stream: {
        ...existing.request.stream,
        ingest: {
          ...existing.request.stream.ingest,
          wired: {
            ...existing.request.stream.ingest.wired,
            routing: mergedRouting,
            fields: mergedFields
          }
        }
      }
    },
    children: mergedChildren
  };
}
function assertNoConflicts(existing, incoming) {
  // routing
  for (const {
    destination
  } of incoming.request.stream.ingest.wired.routing) {
    if (existing.request.stream.ingest.wired.routing.some(rule => rule.destination === destination)) {
      throw new _error.ContentPackConflictError(`[${destination}] already exists`);
    }
  }

  // fields
  for (const [field, fieldConfig] of Object.entries(incoming.request.stream.ingest.wired.fields)) {
    const existingField = existing.request.stream.ingest.wired.fields[field];
    if (existingField && !(0, _lodash.isEqual)(existingField, fieldConfig)) {
      throw new _error.ContentPackConflictError(`Cannot change mapping of [${field}] for [${existing.name}]`);
    }
  }

  // queries
  for (const {
    id,
    title
  } of incoming.request.queries) {
    if (existing.request.queries.some(query => query.id === id)) {
      throw new _error.ContentPackConflictError(`Query [${id} | ${title}] already exists on [${existing.name}]`);
    }
  }
}