"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ActionTypeId = void 0;
exports.executor = executor;
exports.getActionType = getActionType;
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _axios = _interopRequireDefault(require("axios"));
var _configSchema = require("@kbn/config-schema");
var _pipeable = require("fp-ts/pipeable");
var _Option = require("fp-ts/Option");
var _common = require("@kbn/actions-plugin/common");
var _mustache_renderer = require("@kbn/actions-plugin/server/lib/mustache_renderer");
var _axios_utils = require("@kbn/actions-plugin/server/lib/axios_utils");
var _torq = require("../../../common/torq");
var _http_response_retry_header = require("../lib/http_response_retry_header");
var _result_type = require("../lib/result_type");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const configSchemaProps = {
  webhookIntegrationUrl: _configSchema.schema.string()
};
const ConfigSchema = _configSchema.schema.object(configSchemaProps);

// secrets definition

const secretSchemaProps = {
  token: _configSchema.schema.string()
};
const SecretsSchema = _configSchema.schema.object(secretSchemaProps);

// params definition

const ParamsSchema = _configSchema.schema.object({
  body: _configSchema.schema.string()
});
const ActionTypeId = exports.ActionTypeId = '.torq';
// action type definition
function getActionType() {
  return {
    id: ActionTypeId,
    minimumLicenseRequired: 'gold',
    name: _i18n.i18n.translate('xpack.stackConnectors.torqTitle', {
      defaultMessage: 'Torq'
    }),
    supportedFeatureIds: [_common.AlertingConnectorFeatureId, _common.UptimeConnectorFeatureId, _common.SecurityConnectorFeatureId],
    validate: {
      config: {
        schema: _configSchema.schema.object(configSchemaProps),
        customValidator: validateActionTypeConfig
      },
      secrets: {
        schema: SecretsSchema
      },
      params: {
        schema: ParamsSchema
      }
    },
    renderParameterTemplates,
    executor: (0, _lodash.curry)(executor)()
  };
}
function renderParameterTemplates(logger, params, variables) {
  if (!params.body) return params;
  return (0, _mustache_renderer.renderMustacheObject)(logger, params, variables);
}
function validateActionTypeConfig(configObject, validatorServices) {
  const configuredUrl = configObject.webhookIntegrationUrl;
  let configureUrlObj;
  try {
    configureUrlObj = new URL(configuredUrl);
  } catch (err) {
    throw new Error(_i18n.i18n.translate('xpack.stackConnectors.torq.torqConfigurationErrorNoHostname', {
      defaultMessage: 'error configuring send to Torq action: unable to parse url: {err}',
      values: {
        err: err.toString()
      }
    }));
  }
  try {
    validatorServices.configurationUtilities.ensureUriAllowed(configuredUrl);
  } catch (allowListError) {
    throw new Error(_i18n.i18n.translate('xpack.stackConnectors.torq.torqConfigurationError', {
      defaultMessage: 'error configuring send to Torq action: {message}',
      values: {
        message: allowListError.message
      }
    }));
  }
  if (!(0, _torq.isValidTorqHostName)(configureUrlObj.hostname) && configureUrlObj.hostname !== 'localhost') {
    throw new Error(_i18n.i18n.translate('xpack.stackConnectors.torq.torqConfigurationErrorInvalidHostname', {
      defaultMessage: 'error configuring send to Torq action: url must begin with https://hooks.torq.io or https://hooks.eu.torq.io'
    }));
  }
}

// action executor
async function executor(execOptions) {
  const actionId = execOptions.actionId;
  const {
    webhookIntegrationUrl
  } = execOptions.config;
  const {
    body: data
  } = execOptions.params;
  const configurationUtilities = execOptions.configurationUtilities;
  const connectorUsageCollector = execOptions.connectorUsageCollector;
  const secrets = execOptions.secrets;
  const token = secrets.token;
  let body;
  try {
    body = JSON.parse(data || 'null');
  } catch (err) {
    return errorInvalidBody(actionId, execOptions.logger, err);
  }
  const axiosInstance = _axios.default.create();
  const result = await (0, _result_type.promiseResult)((0, _axios_utils.request)({
    axios: axiosInstance,
    url: webhookIntegrationUrl,
    method: 'post',
    headers: {
      'X-Torq-Token': token || '',
      'Content-Type': 'application/json'
    },
    data: body,
    configurationUtilities,
    logger: execOptions.logger,
    validateStatus: status => status >= 200 && status < 300,
    connectorUsageCollector
  }));
  if ((0, _result_type.isOk)(result)) {
    const {
      value: {
        status,
        statusText
      }
    } = result;
    execOptions.logger.debug(`response from Torq action "${actionId}": [HTTP ${status}] ${statusText}`);
    return successResult(actionId, data);
  }
  const {
    error
  } = result;
  return handleExecutionError(error, execOptions.logger, actionId);
}
async function handleExecutionError(error, logger, actionId) {
  if (error.response) {
    const {
      status,
      statusText,
      headers: responseHeaders,
      data: {
        message: responseMessage
      }
    } = error.response;
    const responseMessageAsSuffix = responseMessage ? `: ${responseMessage}` : '';
    const message = `[${status}] ${statusText}${responseMessageAsSuffix}`;
    logger.error(`error on ${actionId} Torq event: ${message}`);
    // The request was made and the server responded with a status code
    // that falls out of the range of 2xx
    // special handling for 5xx
    if (status >= 500) {
      return retryResult(actionId, message);
    }

    // special handling for rate limiting
    if (status === 429) {
      return (0, _pipeable.pipe)((0, _http_response_retry_header.getRetryAfterIntervalFromHeaders)(responseHeaders), (0, _Option.map)(retry => retryResultSeconds(actionId, message, retry)), (0, _Option.getOrElse)(() => retryResult(actionId, message)));
    }
    if (status === 405) {
      return errorResultInvalidMethod(actionId, message);
    }
    if (status === 401) {
      return errorResultUnauthorised(actionId, message);
    }
    if (status === 404) {
      return errorNotFound(actionId, message);
    }
    return errorResultInvalid(actionId, message);
  } else if (error.code) {
    const message = `[${error.code}] ${error.message}`;
    logger.error(`error on ${actionId} Torq event: ${message}`);
    return errorResultRequestFailed(actionId, message);
  } else if (error.isAxiosError) {
    const message = `${error.message}`;
    logger.error(`error on ${actionId} Torq event: ${message}`);
    return errorResultRequestFailed(actionId, message);
  }
  logger.error(`error on ${actionId} Torq action: unexpected error`);
  return errorResultUnexpectedError(actionId);
}
function successResult(actionId, data) {
  return {
    status: 'ok',
    data,
    actionId
  };
}
function errorInvalidBody(actionId, logger, err) {
  const errMessage = _i18n.i18n.translate('xpack.stackConnectors.torq.invalidBodyErrorMessage', {
    defaultMessage: 'error triggering Torq workflow, invalid body'
  });
  logger.error(`error on ${actionId} Torq event: ${errMessage}: ${err.message}`);
  return {
    status: 'error',
    message: errMessage,
    actionId,
    serviceMessage: err.message
  };
}
function errorResultInvalid(actionId, serviceMessage) {
  const errMessage = _i18n.i18n.translate('xpack.stackConnectors.torq.invalidResponseErrorMessage', {
    defaultMessage: 'error triggering Torq workflow, invalid response'
  });
  return {
    status: 'error',
    message: errMessage,
    actionId,
    serviceMessage
  };
}
function errorNotFound(actionId, serviceMessage) {
  const errMessage = _i18n.i18n.translate('xpack.stackConnectors.torq.notFoundErrorMessage', {
    defaultMessage: 'error triggering Torq workflow, make sure the webhook URL is valid'
  });
  return {
    status: 'error',
    message: errMessage,
    actionId,
    serviceMessage
  };
}
function errorResultRequestFailed(actionId, serviceMessage) {
  const errMessage = _i18n.i18n.translate('xpack.stackConnectors.torq.requestFailedErrorMessage', {
    defaultMessage: 'error triggering Torq workflow, request failed'
  });
  return {
    status: 'error',
    message: errMessage,
    actionId,
    serviceMessage
  };
}
function errorResultInvalidMethod(actionId, serviceMessage) {
  const errMessage = _i18n.i18n.translate('xpack.stackConnectors.torq.invalidMethodErrorMessage', {
    defaultMessage: 'error triggering Torq workflow, method is not supported'
  });
  return {
    status: 'error',
    message: errMessage,
    actionId,
    serviceMessage
  };
}
function errorResultUnauthorised(actionId, serviceMessage) {
  const errMessage = _i18n.i18n.translate('xpack.stackConnectors.torq.unauthorisedErrorMessage', {
    defaultMessage: 'error triggering Torq workflow, unauthorised'
  });
  return {
    status: 'error',
    message: errMessage,
    actionId,
    serviceMessage
  };
}
function errorResultUnexpectedError(actionId) {
  const errMessage = _i18n.i18n.translate('xpack.stackConnectors.torq.unreachableErrorMessage', {
    defaultMessage: 'error triggering Torq workflow, unexpected error'
  });
  return {
    status: 'error',
    message: errMessage,
    actionId
  };
}
function retryResult(actionId, serviceMessage) {
  const errMessage = _i18n.i18n.translate('xpack.stackConnectors.torq.invalidResponseRetryLaterErrorMessage', {
    defaultMessage: 'error triggering Torq workflow, retry later'
  });
  return {
    status: 'error',
    message: errMessage,
    retry: true,
    actionId,
    serviceMessage
  };
}
function retryResultSeconds(actionId, serviceMessage, retryAfter) {
  const retryEpoch = Date.now() + retryAfter * 1000;
  const retry = new Date(retryEpoch);
  const retryString = retry.toISOString();
  const errMessage = _i18n.i18n.translate('xpack.stackConnectors.torq.invalidResponseRetryDateErrorMessage', {
    defaultMessage: 'error triggering Torq workflow, retry at {retryString}',
    values: {
      retryString
    }
  });
  return {
    status: 'error',
    message: errMessage,
    retry,
    actionId,
    serviceMessage
  };
}