"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.extractEntityAndBoundaryReferences = extractEntityAndBoundaryReferences;
exports.getRuleType = getRuleType;
exports.injectEntityAndBoundaryIds = injectEntityAndBoundaryIds;
var _i18n = require("@kbn/i18n");
var _server = require("@kbn/core/server");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _geo_containment = require("@kbn/response-ops-rule-params/geo_containment");
var _executor = require("./executor");
var _constants = require("./constants");
var _constants2 = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const actionVariables = {
  context: [
  // Alert-specific data
  {
    name: 'entityId',
    description: _i18n.i18n.translate('xpack.stackAlerts.geoContainment.actionVariableContextEntityIdLabel', {
      defaultMessage: 'The entity ID of the document that triggered the alert'
    })
  }, {
    name: 'entityDateTime',
    description: _i18n.i18n.translate('xpack.stackAlerts.geoContainment.actionVariableContextFromEntityDateTimeLabel', {
      defaultMessage: `The date the entity was recorded in the boundary`
    })
  }, {
    name: 'entityDocumentId',
    description: _i18n.i18n.translate('xpack.stackAlerts.geoContainment.actionVariableContextFromEntityDocumentIdLabel', {
      defaultMessage: 'The ID of the contained entity document'
    })
  }, {
    name: 'detectionDateTime',
    description: _i18n.i18n.translate('xpack.stackAlerts.geoContainment.actionVariableContextDetectionDateTimeLabel', {
      defaultMessage: 'The end of the check interval when the alert occurred'
    })
  }, {
    name: 'entityLocation',
    description: _i18n.i18n.translate('xpack.stackAlerts.geoContainment.actionVariableContextFromEntityLocationLabel', {
      defaultMessage: 'The location of the entity'
    })
  }, {
    name: 'containingBoundaryId',
    description: _i18n.i18n.translate('xpack.stackAlerts.geoContainment.actionVariableContextContainingBoundaryIdLabel', {
      defaultMessage: 'The ID of the boundary containing the entity. Value not set for recovered alerts'
    })
  }, {
    name: 'containingBoundaryName',
    description: _i18n.i18n.translate('xpack.stackAlerts.geoContainment.actionVariableContextContainingBoundaryNameLabel', {
      defaultMessage: 'The name of the boundary containing the entity. Value not set for recovered alerts'
    })
  }]
};
function extractEntityAndBoundaryReferences(params) {
  const {
    indexId,
    boundaryIndexId,
    ...otherParams
  } = params;

  //  Reference names omit the `param:`-prefix. This is handled by the alerting framework already
  const references = [{
    name: `tracked_index_${indexId}`,
    type: 'index-pattern',
    id: indexId
  }, {
    name: `boundary_index_${boundaryIndexId}`,
    type: 'index-pattern',
    id: boundaryIndexId
  }];
  return {
    params: {
      ...otherParams,
      indexRefName: `tracked_index_${indexId}`,
      boundaryIndexRefName: `boundary_index_${boundaryIndexId}`
    },
    references
  };
}
function injectEntityAndBoundaryIds(params, references) {
  const {
    indexRefName,
    boundaryIndexRefName,
    ...otherParams
  } = params;
  const {
    id: indexId = null
  } = references.find(ref => ref.name === indexRefName) || {};
  const {
    id: boundaryIndexId = null
  } = references.find(ref => ref.name === boundaryIndexRefName) || {};
  if (!indexId) {
    throw new Error(`Index "${indexId}" not found in references array`);
  }
  if (!boundaryIndexId) {
    throw new Error(`Boundary index "${boundaryIndexId}" not found in references array`);
  }
  return {
    ...otherParams,
    indexId,
    boundaryIndexId
  };
}
function getRuleType() {
  const alertTypeName = _i18n.i18n.translate('xpack.stackAlerts.geoContainment.alertTypeTitle', {
    defaultMessage: 'Tracking containment'
  });
  const actionGroupName = _i18n.i18n.translate('xpack.stackAlerts.geoContainment.actionGroupContainmentMetTitle', {
    defaultMessage: 'Tracking containment met'
  });
  return {
    id: _constants.GEO_CONTAINMENT_ID,
    name: alertTypeName,
    actionGroups: [{
      id: _constants.ActionGroupId,
      name: actionGroupName
    }],
    recoveryActionGroup: {
      id: _constants.RecoveryActionGroupId,
      name: _i18n.i18n.translate('xpack.stackAlerts.geoContainment.notGeoContained', {
        defaultMessage: 'No longer contained'
      })
    },
    doesSetRecoveryContext: true,
    defaultActionGroupId: _constants.ActionGroupId,
    executor: _executor.executor,
    category: _server.DEFAULT_APP_CATEGORIES.management.id,
    producer: _ruleDataUtils.STACK_ALERTS_FEATURE_ID,
    solution: 'stack',
    validate: {
      params: _geo_containment.trackingContainmentRuleParamsSchema
    },
    schemas: {
      params: {
        type: 'config-schema',
        schema: _geo_containment.trackingContainmentRuleParamsSchema
      }
    },
    actionVariables,
    minimumLicenseRequired: 'gold',
    isExportable: true,
    useSavedObjectReferences: {
      extractReferences: params => {
        return extractEntityAndBoundaryReferences(params);
      },
      injectReferences: (params, references) => {
        return injectEntityAndBoundaryIds(params, references);
      }
    },
    alerts: _constants2.STACK_ALERTS_AAD_CONFIG
  };
}