"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initGetSpaceContentSummaryApi = initGetSpaceContentSummaryApi;
var _lodash = require("lodash");
var _configSchema = require("@kbn/config-schema");
var _server = require("@kbn/core/server");
var _errors = require("../../../lib/errors");
var _space_schema = require("../../../lib/space_schema");
var _lib = require("../../lib");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function initGetSpaceContentSummaryApi(deps) {
  const {
    router,
    getSpacesService
  } = deps;
  router.get({
    path: '/internal/spaces/{spaceId}/content_summary',
    security: {
      authz: {
        requiredPrivileges: ['manage_spaces']
      }
    },
    validate: {
      params: _configSchema.schema.object({
        spaceId: _configSchema.schema.string({
          validate: value => {
            if (!_space_schema.SPACE_ID_REGEX.test(value)) {
              return `lower case, a-z, 0-9, "_", and "-" are allowed.`;
            }
          },
          minLength: 1
        })
      })
    }
  }, (0, _lib.createLicensedRouteHandler)(async (context, request, response) => {
    try {
      var _data$aggregations;
      const spaceId = request.params.spaceId;
      const spacesClient = getSpacesService().createSpacesClient(request);
      await spacesClient.get(spaceId);
      const {
        getClient,
        typeRegistry
      } = (await context.core).savedObjects;
      const client = getClient();
      const types = typeRegistry.getImportableAndExportableTypes().filter(type => !typeRegistry.isNamespaceAgnostic(type.name));
      const searchTypeNames = types.map(type => type.name);
      const data = await client.find({
        type: searchTypeNames,
        perPage: 0,
        namespaces: [spaceId],
        aggs: {
          typesAggregation: {
            terms: {
              field: 'type',
              size: types.length
            }
          }
        }
      });
      const typesMetaInfo = types.reduce((acc, currentType) => {
        var _currentType$manageme, _currentType$manageme2, _currentType$manageme3;
        acc[currentType.name] = {
          displayName: (_currentType$manageme = (_currentType$manageme2 = currentType.management) === null || _currentType$manageme2 === void 0 ? void 0 : _currentType$manageme2.displayName) !== null && _currentType$manageme !== void 0 ? _currentType$manageme : (0, _lodash.capitalize)(currentType.name),
          icon: (_currentType$manageme3 = currentType.management) === null || _currentType$manageme3 === void 0 ? void 0 : _currentType$manageme3.icon
        };
        return acc;
      }, {});
      const summary = (0, _lodash.sortBy)((_data$aggregations = data.aggregations) === null || _data$aggregations === void 0 ? void 0 : _data$aggregations.typesAggregation.buckets.map(item => ({
        count: item.doc_count,
        type: item.key,
        ...typesMetaInfo[item.key]
      })), item => item.displayName.toLowerCase());
      return response.ok({
        body: {
          summary,
          total: data.total
        }
      });
    } catch (error) {
      if (_server.SavedObjectsErrorHelpers.isNotFoundError(error)) {
        return response.notFound();
      }
      return response.customError((0, _errors.wrapError)(error));
    }
  }));
}