"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ApmTransactionErrorRateTransformGenerator = void 0;
var _sloSchema = require("@kbn/slo-schema");
var _ = require(".");
var _constants = require("../../../common/constants");
var _slo_transform_template = require("../../assets/transform_templates/slo_transform_template");
var _errors = require("../../errors");
var _common = require("./common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class ApmTransactionErrorRateTransformGenerator extends _.TransformGenerator {
  constructor(spaceId, dataViewService, isServerless) {
    super(spaceId, dataViewService, isServerless);
  }
  async getTransformParams(slo) {
    if (!_sloSchema.apmTransactionErrorRateIndicatorSchema.is(slo.indicator)) {
      throw new _errors.InvalidTransformError(`Cannot handle SLO of indicator type: ${slo.indicator.type}`);
    }
    return (0, _slo_transform_template.getSLOTransformTemplate)(this.buildTransformId(slo), this.buildDescription(slo), await this.buildSource(slo, slo.indicator), this.buildDestination(slo), this.buildGroupBy(slo, slo.indicator), this.buildAggregations(slo), this.buildSettings(slo, '@timestamp'), slo);
  }
  buildTransformId(slo) {
    return (0, _constants.getSLOTransformId)(slo.id, slo.revision);
  }
  buildGroupBy(slo, indicator) {
    // These groupBy fields must match the fields from the source query, otherwise
    // the transform will create permutations for each value present in the source.
    // E.g. if environment is not specified in the source query, but we include it in the groupBy,
    // we'll output documents for each environment value
    const extraGroupByFields = {
      ...(indicator.params.service !== _sloSchema.ALL_VALUE && {
        'service.name': {
          terms: {
            field: 'service.name'
          }
        }
      }),
      ...(indicator.params.environment !== _sloSchema.ALL_VALUE && {
        'service.environment': {
          terms: {
            field: 'service.environment'
          }
        }
      }),
      ...(indicator.params.transactionName !== _sloSchema.ALL_VALUE && {
        'transaction.name': {
          terms: {
            field: 'transaction.name'
          }
        }
      }),
      ...(indicator.params.transactionType !== _sloSchema.ALL_VALUE && {
        'transaction.type': {
          terms: {
            field: 'transaction.type'
          }
        }
      })
    };
    return this.buildCommonGroupBy(slo, '@timestamp', extraGroupByFields);
  }
  async buildSource(slo, indicator) {
    const queryFilter = [(0, _common.getFilterRange)(slo, '@timestamp', this.isServerless)];
    if (indicator.params.service !== _sloSchema.ALL_VALUE) {
      queryFilter.push({
        match: {
          'service.name': indicator.params.service
        }
      });
    }
    if (indicator.params.environment !== _sloSchema.ALL_VALUE) {
      queryFilter.push({
        match: {
          'service.environment': indicator.params.environment
        }
      });
    }
    if (indicator.params.transactionName !== _sloSchema.ALL_VALUE) {
      queryFilter.push({
        match: {
          'transaction.name': indicator.params.transactionName
        }
      });
    }
    if (indicator.params.transactionType !== _sloSchema.ALL_VALUE) {
      queryFilter.push({
        match: {
          'transaction.type': indicator.params.transactionType
        }
      });
    }
    const dataView = await this.getIndicatorDataView(indicator.params.dataViewId);
    if (indicator.params.filter) {
      queryFilter.push((0, _.getElasticsearchQueryOrThrow)(indicator.params.filter, dataView));
    }
    return {
      index: (0, _common.parseIndex)(indicator.params.index),
      runtime_mappings: this.buildCommonRuntimeMappings(dataView),
      query: {
        bool: {
          filter: [{
            term: {
              'metricset.name': 'transaction'
            }
          }, {
            terms: {
              'event.outcome': ['success', 'failure']
            }
          }, ...queryFilter]
        }
      }
    };
  }
  buildDestination(slo) {
    return {
      pipeline: (0, _constants.getSLOPipelineId)(slo.id, slo.revision),
      index: _constants.SLI_DESTINATION_INDEX_NAME
    };
  }
  buildAggregations(slo) {
    return {
      'slo.numerator': {
        filter: {
          bool: {
            should: {
              match: {
                'event.outcome': 'success'
              }
            }
          }
        }
      },
      'slo.denominator': {
        filter: {
          match_all: {}
        }
      },
      ...(_sloSchema.timeslicesBudgetingMethodSchema.is(slo.budgetingMethod) && {
        'slo.isGoodSlice': {
          bucket_script: {
            buckets_path: {
              goodEvents: 'slo.numerator>_count',
              totalEvents: 'slo.denominator>_count'
            },
            script: `if (params.totalEvents == 0) { return 1 } else { return params.goodEvents / params.totalEvents ${(0, _common.getTimesliceTargetComparator)(slo.objective.timesliceTarget)} ${slo.objective.timesliceTarget} ? 1 : 0 }`
          }
        }
      })
    };
  }
}
exports.ApmTransactionErrorRateTransformGenerator = ApmTransactionErrorRateTransformGenerator;